/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.wx.service.service.impl;

import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.smart.wx.core.exception.ValidationFileContentInvalidException;
import net.guerlab.smart.wx.core.exception.ValidationFileNameHasBlankErrorException;
import net.guerlab.smart.wx.core.exception.ValidationFileNameInvalidException;
import net.guerlab.smart.wx.core.exception.ValidationFileNameSuffixErrorException;
import net.guerlab.smart.wx.core.searchparams.ValidationFileSearchParams;
import net.guerlab.smart.wx.service.entity.ValidationFile;
import net.guerlab.smart.wx.service.mapper.ValidationFileMapper;
import net.guerlab.smart.wx.service.service.ValidationFileService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

/**
 * 验证文件服务实现
 *
 * @author guer
 */
@Service
public class ValidationFileServiceImpl extends BaseServiceImpl<ValidationFile, String, ValidationFileMapper, ValidationFileSearchParams>
        implements ValidationFileService {

    @Override
    public void insert(ValidationFile entity) {
        insertBefore(entity);
        getBaseMapper().replaceInsert(entity);
    }

    @Override
    protected void insertBefore(ValidationFile entity) {
        String fileName = StringUtils.trimToNull(entity.getFileName());
        String content = StringUtils.trimToNull(entity.getContent());

        if (fileName == null) {
            throw new ValidationFileNameInvalidException();
        } else if (hasBlankCheck(fileName)) {
            throw new ValidationFileNameHasBlankErrorException();
        } else if (!fileName.endsWith(FILE_SUFFIX)) {
            throw new ValidationFileNameSuffixErrorException();
        }
        if (content == null) {
            throw new ValidationFileContentInvalidException();
        }

        entity.setFileName(fileName.replace(FILE_SUFFIX, ""));
        entity.setContent(content);
    }

    private boolean hasBlankCheck(final CharSequence cs) {
        int strLen;
        if (cs == null || (strLen = cs.length()) == 0) {
            return true;
        }
        for (int i = 0; i < strLen; i++) {
            if (Character.isWhitespace(cs.charAt(i))) {
                return true;
            }
        }
        return false;
    }

    @Override
    protected void updateBefore(ValidationFile entity) {
        entity.setContent(StringUtils.trimToNull(entity.getContent()));
    }
}
