package net.guerlab.smart.wx.service.service.impl;

import lombok.extern.slf4j.Slf4j;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.smart.wx.core.searchparams.WxUserLoginLogSearchParams;
import net.guerlab.smart.wx.service.entity.LoginStatistics;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.entity.WxUserLoginLog;
import net.guerlab.smart.wx.service.handler.AfterWxUserUpdateHandler;
import net.guerlab.smart.wx.service.mapper.WxUserLoginLogMapper;
import net.guerlab.smart.wx.service.service.WxUserLoginLogService;
import net.guerlab.smart.wx.stream.binders.WxAppChangeMessage;
import net.guerlab.smart.wx.stream.binders.WxAppChangeSubscriberChannel;
import net.guerlab.spring.searchparams.SearchParamsUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.messaging.Message;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * 微信用户登录记录服务实现
 *
 * @author guer
 */
@Slf4j
@Service
@EnableBinding({ WxAppChangeSubscriberChannel.class })
public class WxUserLoginLogServiceImpl extends BaseServiceImpl<WxUserLoginLog, Long, WxUserLoginLogMapper>
        implements WxUserLoginLogService, AfterWxUserUpdateHandler {

    @Override
    public void addLog(WxUser user, String ip, String referer) {
        if (user == null) {
            return;
        }

        WxUserLoginLog log = new WxUserLoginLog();
        log.setLoginLogId(sequence.nextId());
        log.setOpenId(user.getOpenId());
        log.setAppId(user.getAppId());
        log.setAppName(user.getAppName());
        log.setUnionId(user.getUnionId());
        log.setAvatarUrl(user.getAvatarUrl());
        log.setNickName(user.getNickName());
        log.setActivated(user.getActivated());
        log.setLoginTime(LocalDateTime.now());
        log.setIp(StringUtils.trimToEmpty(ip));
        log.setReferer(StringUtils.trimToEmpty(referer));

        mapper.insertSelective(log);
    }

    @Override
    public Collection<LoginStatistics> selectLoginStatistics(WxUserLoginLogSearchParams searchParams) {
        Map<String, Object> params = new HashMap<>(16);
        SearchParamsUtils.handler(searchParams, params);
        try {
            return mapper.selectLoginStatistics(params);
        } catch (Exception e) {
            log.debug(e.getLocalizedMessage(), e);
            return Collections.emptyList();
        }
    }

    @Override
    public void afterWxUserUpdateHandler(WxUser wxUser) {
        String openId = StringUtils.trimToNull(wxUser.getOpenId());
        String unionId = StringUtils.trimToEmpty(wxUser.getUnionId());
        String avatarUrl = StringUtils.trimToNull(wxUser.getAvatarUrl());
        String nickName = StringUtils.trimToNull(wxUser.getNickName());
        boolean noUpdate = avatarUrl == null && nickName == null;

        if (openId == null || noUpdate) {
            return;
        }

        WxUserLoginLog update = new WxUserLoginLog();
        update.setUnionId(unionId);
        update.setAvatarUrl(avatarUrl);
        update.setNickName(nickName);

        WxUserLoginLogSearchParams searchParams = new WxUserLoginLogSearchParams();
        searchParams.setOpenId(openId);

        mapper.updateByExampleSelective(update, getExample(searchParams));
    }

    @StreamListener(WxAppChangeSubscriberChannel.NAME)
    public void wxAppChangeMessageHandler(Message<WxAppChangeMessage> message) {
        WxAppChangeMessage payload = message.getPayload();

        if (payload.isDeleteFlag()) {
            return;
        }

        String appId = StringUtils.trimToNull(payload.getAppId());
        String appName = payload.getApp() == null ? null : StringUtils.trimToNull(payload.getApp().getAppName());

        if (appId == null || appName == null) {
            return;
        }

        WxUserLoginLog update = new WxUserLoginLog();
        update.setAppName(appName);

        WxUserLoginLogSearchParams searchParams = new WxUserLoginLogSearchParams();
        searchParams.setAppId(appId);

        mapper.updateByExampleSelective(update, getExample(searchParams));
    }
}
