package net.guerlab.smart.wx.service.controller;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.smart.wx.core.message.AbstractRequest;
import net.guerlab.smart.wx.core.message.SendMsgResponse;
import net.guerlab.smart.wx.core.searchparams.WxUserSearchParams;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.service.WxUserService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 抽象消息发送
 *
 * @author guer
 */
public abstract class AbstractMsgController<Q extends AbstractRequest<?>> {

    protected WxUserService wxUserService;

    @ApiOperation("发送消息")
    @PostMapping("/{appId}/sendMsg")
    public Map<String, SendMsgResponse> sendMsg(@ApiParam(value = "应用ID", required = true) @PathVariable String appId,
            @RequestBody Q request) {
        Collection<String> openIds = getOpenIds(appId, request);
        if (openIds.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<String, SendMsgResponse> result = sendMsg0(appId, openIds, request);
        afterSendMsg(appId, openIds, request);
        return result;
    }

    /**
     * 发送后置
     *
     * @param appId
     *         应用ID
     * @param openIds
     *         openId列表
     * @param request
     *         发送请求
     */
    protected void afterSendMsg(String appId, Collection<String> openIds, Q request) {

    }

    /**
     * 过滤openId
     *
     * @param appId
     *         应用ID
     * @param request
     *         发送请求
     * @return 过滤后的openId列表
     */
    protected Collection<String> getOpenIds(String appId, Q request) {
        boolean selectAllUser = request.getSelectAllUser() != null && request.getSelectAllUser();

        Collection<String> result;
        if (selectAllUser) {
            WxUserSearchParams searchParams = request.getSearchParams();
            if (searchParams == null) {
                searchParams = new WxUserSearchParams();
            }

            searchParams.setAppId(appId);
            result = CollectionUtil.toList(wxUserService.queryAll(searchParams), WxUser::getOpenId);
        } else {
            result = request.getOpenIds();
        }

        if (result == null) {
            return Collections.emptyList();
        }

        return result.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
    }

    /**
     * 发送实现
     *
     * @param appId
     *         应用ID
     * @param openIds
     *         openId列表
     * @param request
     *         发送请求
     * @return 发送响应
     */
    protected abstract Map<String, SendMsgResponse> sendMsg0(String appId, Collection<String> openIds, Q request);

    @Autowired
    public void setWxUserService(WxUserService wxUserService) {
        this.wxUserService = wxUserService;
    }
}
