package net.guerlab.smart.wx.service.service.impl;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.api.impl.WxMaServiceImpl;
import cn.binarywang.wx.miniapp.config.WxMaConfig;
import net.guerlab.smart.wx.core.enums.WxAppType;
import net.guerlab.smart.wx.core.exception.WxAppInvalidException;
import net.guerlab.smart.wx.miniapp.spring.storate.WxMiniAppRedisTemplateConfigStorage;
import net.guerlab.smart.wx.service.entity.WxApp;
import net.guerlab.smart.wx.service.service.WxAppService;
import net.guerlab.smart.wx.service.service.WxMaManagerService;
import net.guerlab.smart.wx.stream.binders.WxAppChangeMessage;
import net.guerlab.smart.wx.stream.binders.WxAppChangeSubscriberChannel;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.messaging.Message;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 微信小程序服务管理实现
 *
 * @author guer
 */
@Service
@EnableBinding({ WxAppChangeSubscriberChannel.class })
public class WxMaManagerServiceImpl implements WxMaManagerService {

    private final Map<String, WxMaService> serviceMap = new ConcurrentHashMap<>(8);

    private final Lock initLock = new ReentrantLock();

    private WxAppService wxAppService;

    private RedisTemplate<String, String> redisTemplate;

    @StreamListener(WxAppChangeSubscriberChannel.NAME)
    public void wxAppChangeMessageHandler(Message<WxAppChangeMessage> message) {
        WxAppChangeMessage payload = message.getPayload();

        initLock.lock();
        try {
            serviceMap.remove(payload.getAppId());
        } finally {
            initLock.unlock();
        }
    }

    @Override
    public WxMaService getService(String appId) {
        appId = StringUtils.trimToNull(appId);
        if (appId == null) {
            return null;
        }

        WxMaService service = serviceMap.get(appId);

        if (service != null) {
            return service;
        }

        initLock.lock();
        try {
            WxMaConfig config = buildConfigStorage(appId);
            service = new WxMaServiceImpl();
            service.setWxMaConfig(config);

            serviceMap.put(appId, service);

            return service;
        } finally {
            initLock.unlock();
        }
    }

    private WxMaConfig buildConfigStorage(String appId) {
        WxApp wxApp = wxAppService.selectById(appId);
        if (wxApp == null || wxApp.getWxAppType() != WxAppType.MINI_APP) {
            throw new WxAppInvalidException();
        }

        WxMiniAppRedisTemplateConfigStorage storage = new WxMiniAppRedisTemplateConfigStorage(redisTemplate);
        storage.setAppid(wxApp.getAppId());
        storage.setSecret(wxApp.getSecret());
        storage.setToken(wxApp.getToken());
        storage.setAesKey(wxApp.getAesKey());
        storage.setMsgDataFormat(wxApp.getMsgDataFormat().toString());
        return storage;
    }

    @Autowired
    public void setWxAppService(WxAppService wxAppService) {
        this.wxAppService = wxAppService;
    }

    @Autowired
    public void setRedisTemplate(RedisTemplate<String, String> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }
}
