package net.guerlab.smart.wx.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.smart.platform.stream.utils.MessageUtils;
import net.guerlab.smart.wx.core.enums.WxAppType;
import net.guerlab.smart.wx.core.enums.WxMsgDataFormat;
import net.guerlab.smart.wx.core.exception.*;
import net.guerlab.smart.wx.core.utils.WxCpUtils;
import net.guerlab.smart.wx.service.entity.WxApp;
import net.guerlab.smart.wx.service.mapper.WxAppMapper;
import net.guerlab.smart.wx.service.service.WxAppService;
import net.guerlab.smart.wx.stream.binders.WxAppChangeMessage;
import net.guerlab.smart.wx.stream.binders.WxAppChangeSenderChannel;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * 微信应用实现
 *
 * @author guer
 */
@Service
@EnableBinding({ WxAppChangeSenderChannel.class })
public class WxAppServiceImpl extends BaseServiceImpl<WxApp, String, WxAppMapper> implements WxAppService {

    private WxAppChangeSenderChannel wxAppChangeSenderChannel;

    @Override
    protected void insertBefore(WxApp entity) {
        String appId = StringUtils.trimToNull(entity.getAppId());
        String corpId = StringUtils.trimToNull(entity.getCorpId());
        Integer agentId = entity.getAgentId();
        String secret = StringUtils.trimToNull(entity.getSecret());
        String appName = StringUtils.trimToNull(entity.getAppName());
        WxMsgDataFormat msgDataFormat = entity.getMsgDataFormat();
        WxAppType appType = entity.getWxAppType();

        if (appType == WxAppType.CP) {
            if (corpId == null) {
                throw new CorpIdInvalidException();
            }
            if (!NumberHelper.greaterZero(agentId)) {
                throw new AgentIdInvalidException();
            }
            appId = WxCpUtils.buildAppId(corpId, agentId);
        } else {
            if (appId == null) {
                throw new AppIdInvalidException();
            }
            corpId = appId;
            agentId = WxApp.DEFAULT_AGENT_ID;
        }
        if (secret == null) {
            throw new SecretInvalidException();
        }
        if (appType == null) {
            throw new WxAppTypeInvalidException();
        }
        if (selectById(appId) != null) {
            throw new AppIdRepeatException();
        }

        LocalDateTime now = LocalDateTime.now();

        entity.setAppId(appId);
        entity.setCorpId(corpId);
        entity.setAgentId(agentId);
        entity.setSecret(secret);
        entity.setToken(StringUtils.trimToEmpty(entity.getToken()));
        entity.setAesKey(StringUtils.trimToEmpty(entity.getAesKey()));
        if (msgDataFormat == null || appType != WxAppType.MINI_APP) {
            entity.setMsgDataFormat(WxMsgDataFormat.XML);
        }
        entity.setAppName(appName != null ? appName : appId);
        if (entity.getEnable() == null) {
            entity.setEnable(true);
        }
        entity.setRemark(StringUtils.trimToEmpty(entity.getRemark()));
        entity.setCreateTime(now);
        entity.setUpdateTime(now);
    }

    @Override
    protected void updateBefore(WxApp entity) {
        if (entity.getWxAppType() != WxAppType.MINI_APP) {
            entity.setMsgDataFormat(null);
        }
    }

    @Override
    protected void updateAfter(WxApp entity) {
        WxAppChangeMessage message = new WxAppChangeMessage();
        message.setAppId(entity.getAppId());
        message.setApp(entity.toDTO());
        message.setDeleteFlag(false);

        notifyChange(message);
    }

    @Override
    protected void deleteByIdAfter(String id, Boolean force) {
        WxAppChangeMessage message = new WxAppChangeMessage();
        message.setAppId(id);
        message.setDeleteFlag(true);

        notifyChange(message);
    }

    @Override
    protected void deleteAfter(WxApp entity, Boolean force) {
        deleteByIdAfter(entity.getAppId(), force);
    }

    private void notifyChange(WxAppChangeMessage message) {
        if (message != null) {
            MessageUtils.send(wxAppChangeSenderChannel.output(), message);
        }
    }

    @Autowired
    public void setWxAppChangeSenderChannel(WxAppChangeSenderChannel wxAppChangeSenderChannel) {
        this.wxAppChangeSenderChannel = wxAppChangeSenderChannel;
    }
}
