/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.wx.internal.controller.inside;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.wx.core.domain.WxUserDTO;
import net.guerlab.smart.wx.core.exception.WxUserInvalidException;
import net.guerlab.smart.wx.core.searchparams.WxUserSearchParams;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.service.WxUserService;
import net.guerlab.web.result.ListObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

/**
 * 微信用户
 *
 * @author guer
 */
@Tag(name = "微信用户")
@RestController("/inside/wxUser")
@RequestMapping("/inside/wxUser")
public class WxUserController {

    private WxUserService service;

    @Operation(description = "根据AppId和OpenId查询用户")
    @GetMapping("/{appId}/{openId}")
    public WxUserDTO findOne(@Parameter(name = "appId", required = true) @PathVariable String appId, @Parameter(name = "openId", required = true) @PathVariable String openId) {
        WxUser wxUser = service.findUser(appId, openId);

        if (wxUser == null) {
            throw new WxUserInvalidException();
        }

        return wxUser.convert();
    }

    @Operation(description = "根据搜索参数查询用户分页列表")
    @PostMapping
    public ListObject<WxUserDTO> findList(@RequestBody WxUserSearchParams searchParams) {
        return BeanConvertUtils.toListObject(service.queryPage(searchParams));
    }

    @Operation(description = "根据搜索参数查询全部用户列表")
    @PostMapping("/all")
    public Collection<WxUserDTO> findAll(@RequestBody WxUserSearchParams searchParams) {
        return BeanConvertUtils.toList(service.queryAll(searchParams));
    }

    @Operation(description = "根据搜素参数查询用户表")
    @PostMapping("/findMap")
    public Map<String, WxUserDTO> findMap(@RequestBody Collection<String> openIds) {
        if (openIds == null || openIds.isEmpty()) {
            return Collections.emptyMap();
        }

        WxUserSearchParams searchParams = new WxUserSearchParams();
        searchParams.setOpenIds(openIds);

        Collection<WxUser> wxUsers = service.selectAll(searchParams);

        return CollectionUtil.toMap(wxUsers, WxUser::getOpenId, WxUser::convert);
    }

    @Operation(description = "根据搜索参数查询用户数量")
    @PostMapping("/count")
    public int findCount(WxUserSearchParams searchParams) {
        return service.selectCount(searchParams);
    }

    @Autowired
    public void setService(WxUserService service) {
        this.service = service;
    }
}
