/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.wx.auth.factory;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtBuilder;
import net.guerlab.smart.platform.auth.factory.AbstractJwtTokenFactory;
import net.guerlab.smart.wx.auth.properties.WxUserJwtTokenFactoryProperties;
import net.guerlab.smart.wx.core.WxUserAuthConstants;
import net.guerlab.smart.wx.core.entity.IWxUserTokenInfo;
import net.guerlab.smart.wx.core.entity.WxUserTokenInfo;
import org.apache.commons.lang3.StringUtils;

/**
 * jwt token 工厂
 *
 * @author guer
 */
public class WxUserJwtTokenFactory extends AbstractJwtTokenFactory<IWxUserTokenInfo, WxUserJwtTokenFactoryProperties> {

    /**
     * 签名前缀
     */
    public static final String PREFIX = "WX_JWT";

    @Override
    protected void generateToken0(JwtBuilder builder, IWxUserTokenInfo user) {
        Boolean activated = user.getActivated();

        builder.setSubject(user.getNickName());
        builder.claim(WxUserAuthConstants.OPEN_ID, StringUtils.trimToEmpty(user.getOpenId()));
        builder.claim(WxUserAuthConstants.UNION_ID, StringUtils.trimToEmpty(user.getUnionId()));
        builder.claim(WxUserAuthConstants.APP_ID, StringUtils.trimToEmpty(user.getAppId()));
        builder.claim(WxUserAuthConstants.AVATAR_URL, StringUtils.trimToEmpty(user.getAvatarUrl()));
        builder.claim(WxUserAuthConstants.ACTIVATED, activated != null && activated);
    }

    @Override
    protected IWxUserTokenInfo parse0(Claims body) {
        String openId = body.get(WxUserAuthConstants.OPEN_ID, String.class);
        String unionId = body.get(WxUserAuthConstants.UNION_ID, String.class);
        String appId = body.get(WxUserAuthConstants.APP_ID, String.class);
        String nickName = body.getSubject();
        String avatarUrl = body.get(WxUserAuthConstants.AVATAR_URL, String.class);
        Boolean activated = body.get(WxUserAuthConstants.ACTIVATED, Boolean.class);

        WxUserTokenInfo jwtInfo = new WxUserTokenInfo();
        jwtInfo.setOpenId(openId);
        jwtInfo.setUnionId(unionId);
        jwtInfo.setAppId(appId);
        jwtInfo.setNickName(nickName);
        jwtInfo.setAvatarUrl(avatarUrl);
        jwtInfo.setActivated(activated);

        return jwtInfo;
    }

    @Override
    protected String getPrefix() {
        return PREFIX;
    }

    @Override
    public Class<IWxUserTokenInfo> getAcceptClass() {
        return IWxUserTokenInfo.class;
    }
}
