/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.wx.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.wx.api.WxMaApi;
import net.guerlab.smart.wx.api.feign.FeignWxMaApi;
import net.guerlab.smart.wx.core.domain.LoginResponse;
import net.guerlab.smart.wx.core.domain.MaEncryptedData;
import net.guerlab.web.result.Result;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * @author guer
 */
@Configuration
@AutoConfigureAfter(WxMaApiLocalServiceAutoConfigure.class)
public class WxMaApiFeignAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnMissingBean(WxMaApi.class)
    public WxMaApi wxMaApiFeignWrapper(FeignWxMaApi api) {
        return new WxMaApiFeignWrapper(api);
    }

    @AllArgsConstructor
    private static class WxMaApiFeignWrapper implements WxMaApi {

        private final FeignWxMaApi api;

        @Override
        public LoginResponse login(String appId, String code) {
            Result<LoginResponse> result = api.login(appId, code);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public LoginResponse register(String appId, String openId, MaEncryptedData encryptedData) {
            Result<LoginResponse> result = api.register(appId, openId, encryptedData);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public String getAccessToken(String appId) {
            Result<String> result = api.getAccessToken(appId);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public String getPhoneNumber(String appId, String openId, MaEncryptedData encryptedData) {
            Result<String> result = api.getPhoneNumber(appId, openId, encryptedData);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }
    }

}
