package net.guerlab.smart.user.api.feign;

import net.guerlab.smart.user.api.feign.factory.FeignUserApiFallbackFactory;
import net.guerlab.smart.user.core.domain.PositionDataDTO;
import net.guerlab.smart.user.core.domain.UserDTO;
import net.guerlab.smart.user.core.domain.UserModifyDTO;
import net.guerlab.smart.user.core.entity.LoginResponse;
import net.guerlab.smart.user.core.entity.PasswordRequest;
import net.guerlab.smart.user.core.entity.UserInternalLoginRequest;
import net.guerlab.web.result.ListObject;
import net.guerlab.web.result.Result;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用户服务接口
 *
 * @author guer
 */
@FeignClient(name = "user-internal/inside/user", fallbackFactory = FeignUserApiFallbackFactory.class)
public interface FeignUserApi {

    /**
     * 根据用户id查询用户
     *
     * @param userId
     *         用户id
     * @return 用户
     */
    @GetMapping("/{userId}")
    Result<UserDTO> findOne(@PathVariable("userId") Long userId);

    /**
     * 根据用户名查询用户
     *
     * @param username
     *         用户名
     * @return 用户
     */
    @GetMapping("/findOneByUsername/{username}")
    Result<UserDTO> findOneByUsername(@PathVariable("username") String username);

    /**
     * 根据搜索参数查询用户列表
     *
     * @param searchParams
     *         搜索参数
     * @return 用户列表
     */
    @PostMapping
    Result<ListObject<UserDTO>> findList(@RequestBody Map<String, Object> searchParams);

    /**
     * 根据搜索参数查询用户列表
     *
     * @param searchParams
     *         搜索参数
     * @return 用户列表
     */
    @PostMapping("/all")
    Result<List<UserDTO>> findAll(@RequestBody Map<String, Object> searchParams);

    /**
     * 通过用户ID获取权限关键字列表
     *
     * @param userId
     *         用户id
     * @return 权限关键字列表
     */
    @GetMapping("/{userId}/permissionKeys")
    Result<List<String>> permissionKeys(@PathVariable("userId") Long userId);

    /**
     * 判断用户是否拥有指定权限
     *
     * @param userId
     *         用户ID
     * @param keys
     *         权限key列表
     * @return 是否拥有指定权限
     */
    @PostMapping("/{userId}/hasPermission")
    Result<Boolean> hasPermission(@PathVariable("userId") Long userId, @RequestBody Collection<String> keys);

    /**
     * 通过用户ID获取职务信息列表
     *
     * @param userId
     *         用户id
     * @return 职务信息列表
     */
    @GetMapping("/{userId}/position")
    Result<List<PositionDataDTO>> getPosition(@PathVariable("userId") Long userId);

    /**
     * 通过用户ID获取职务信息关键字列表
     *
     * @param userId
     *         用户id
     * @return 职务信息关键字列表
     */
    @GetMapping("/{userId}/positionKeys")
    Result<Set<String>> getPositionKeys(@PathVariable("userId") Long userId);

    /**
     * 检查密码是否正确
     *
     * @param userId
     *         用户id
     * @param request
     *         密码请求
     * @return 密码是否正确
     */
    @GetMapping("/{userId}/checkPassword")
    Result<Boolean> checkPassword(@PathVariable("userId") Long userId, @RequestBody PasswordRequest request);

    /**
     * 添加用户
     *
     * @param user
     *         用户
     * @return 用户
     */
    @PostMapping("/all")
    Result<UserDTO> add(@RequestBody UserModifyDTO user);

    /**
     * 内部登录
     *
     * @param request
     *         请求
     * @return 登录响应
     */
    @PostMapping("/internalLogin")
    Result<LoginResponse> internalLogin(@RequestBody UserInternalLoginRequest request);
}
