package net.guerlab.smart.user.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.smart.platform.commons.exception.UserInvalidException;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.user.api.UserApi;
import net.guerlab.smart.user.core.domain.PositionDataDTO;
import net.guerlab.smart.user.core.domain.UserDTO;
import net.guerlab.smart.user.core.domain.UserModifyDTO;
import net.guerlab.smart.user.core.entity.LoginResponse;
import net.guerlab.smart.user.core.entity.UserInternalLoginRequest;
import net.guerlab.smart.user.core.exception.LoginTypeInvalidException;
import net.guerlab.smart.user.core.exception.NeedPasswordException;
import net.guerlab.smart.user.core.searchparams.UserSearchParams;
import net.guerlab.smart.user.core.utils.PositionUtils;
import net.guerlab.smart.user.service.entity.User;
import net.guerlab.smart.user.service.handler.PositionGetHandler;
import net.guerlab.smart.user.service.service.*;
import net.guerlab.web.result.ListObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.lang.NonNull;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * @author guer
 */
@Configuration
@Conditional(UserApiLocalServiceAutoConfigure.WrapperCondition.class)
public class UserApiLocalServiceAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnBean(UserService.class)
    public UserApi userApiLocalServiceWrapper(UserService service, PositionGetHandler positionGetHandler,
            PositionService positionService, PermissionCheckService permissionCheckService,
            UserLoginService userLoginService, LoginLogService loginLogService) {
        return new UserApiLocalServiceWrapper(service, positionGetHandler, positionService, permissionCheckService,
                userLoginService, loginLogService);
    }

    @SuppressWarnings("WeakerAccess")
    static class WrapperCondition implements Condition {

        @Override
        public boolean matches(@NonNull ConditionContext context, @NonNull AnnotatedTypeMetadata metadata) {
            try {
                ClassLoader classLoader = WrapperCondition.class.getClassLoader();
                String packageName = "net.guerlab.smart.user.service.service.";
                return classLoader.loadClass(packageName + "UserService") != null
                        && classLoader.loadClass(packageName + "PositionGetHandler") != null
                        && classLoader.loadClass(packageName + "PositionService") != null
                        && classLoader.loadClass(packageName + "PermissionCheckService") != null
                        && classLoader.loadClass(packageName + "UserLoginService") != null
                        && classLoader.loadClass(packageName + "LoginLogService") != null;
            } catch (Exception e) {
                return false;
            }
        }
    }

    @AllArgsConstructor
    private static class UserApiLocalServiceWrapper implements UserApi {

        private final UserService service;

        private final PositionGetHandler positionGetHandler;

        private final PositionService positionService;

        private final PermissionCheckService permissionCheckService;

        private final UserLoginService userLoginService;

        private final LoginLogService loginLogService;

        @Override
        public UserDTO findOne(Long userId) {
            return service.selectByIdOptional(userId).orElseThrow(UserInvalidException::new).toDTO();
        }

        @Override
        public UserDTO findOneByUsername(String username) {
            User user = service.selectByUsername(username);

            if (user == null) {
                throw new UserInvalidException();
            }

            return user.toDTO();
        }

        @Override
        public ListObject<UserDTO> findList(UserSearchParams searchParams) {
            return BeanConvertUtils.toListObject(service.selectPage(searchParams));
        }

        @Override
        public List<UserDTO> findAll(UserSearchParams searchParams) {
            return BeanConvertUtils.toList(service.selectAll(searchParams));
        }

        @Override
        public UserDTO add(UserModifyDTO dto) {
            String password = StringUtils.trimToNull(dto.getPassword());

            if (password == null) {
                throw new NeedPasswordException();
            }

            User user = new User();

            BeanUtils.copyProperties(dto, user);
            user.setAdmin(false);

            service.insertSelective(user);

            return user.toDTO();
        }

        @Override
        public LoginResponse internalLogin(UserInternalLoginRequest request) {
            String loginType = StringUtils.trimToNull(request.getLoginType());

            if (loginType == null) {
                throw new LoginTypeInvalidException();
            }

            String ip = StringUtils.trimToNull(request.getIp());
            if (ip == null) {
                ip = "127.0.0.1";
            }

            User user = service.selectByIdOptional(request.getUserId()).orElseThrow(UserInvalidException::new);

            LoginResponse loginResponse = userLoginService.login(user, null, ip);
            loginLogService.addSuccessLog(user.getUserId(), user.getName(), "USER_ID", loginType, ip);
            return loginResponse;
        }

        @Override
        public List<String> permissionKeys(Long userId) {
            return new ArrayList<>(service.getPermissionKeys(userId));
        }

        @Override
        public boolean hasPermission(Long userId, Collection<String> keys) {
            if (keys == null || keys.isEmpty()) {
                return true;
            }

            return permissionCheckService.acceptByPermissionKeys(userId, keys).isAccept();
        }

        @Override
        public List<PositionDataDTO> getPosition(Long userId) {
            return positionGetHandler.getPosition(userId);
        }

        @Override
        public Set<String> getPositionKeys(Long userId) {
            return PositionUtils.getKeys(positionService.findByUserId(userId));
        }

        @Override
        public boolean checkPassword(Long userId, String password) {
            User user = service.selectById(userId);
            return user != null && !service.checkPasswordError(user, password);
        }
    }
}
