package net.guerlab.smart.user.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.user.api.OauthApi;
import net.guerlab.smart.user.core.domain.OauthDTO;
import net.guerlab.smart.user.core.exception.OauthInvalidException;
import net.guerlab.smart.user.core.searchparams.OauthSearchParams;
import net.guerlab.smart.user.service.service.OauthService;
import net.guerlab.web.result.ListObject;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.lang.NonNull;

import java.util.List;

/**
 * @author guer
 */
@Configuration
@Conditional(OauthApiLocalServiceAutoConfigure.WrapperCondition.class)
public class OauthApiLocalServiceAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnBean(OauthService.class)
    public OauthApi oauthApiLocalServiceWrapper(OauthService service) {
        return new OauthApiLocalServiceWrapper(service);
    }

    @SuppressWarnings("WeakerAccess")
    static class WrapperCondition implements Condition {

        @Override
        public boolean matches(@NonNull ConditionContext context, @NonNull AnnotatedTypeMetadata metadata) {
            try {
                ClassLoader classLoader = WrapperCondition.class.getClassLoader();
                return classLoader.loadClass("net.guerlab.smart.user.service.service.OauthService") != null;
            } catch (Exception e) {
                return false;
            }
        }
    }

    @AllArgsConstructor
    private static class OauthApiLocalServiceWrapper implements OauthApi {

        private final OauthService service;

        @Override
        public OauthDTO findOne(String type, Long userId) {
            return service.getOneOptional(type, userId).orElseThrow(OauthInvalidException::new).toDTO();
        }

        @Override
        public OauthDTO findOne(String type, String thirdPartyId) {
            return service.getOneOptional(type, thirdPartyId).orElseThrow(OauthInvalidException::new).toDTO();
        }

        @Override
        public ListObject<OauthDTO> findList(OauthSearchParams searchParams) {
            return BeanConvertUtils.toListObject(service.selectPage(searchParams));
        }

        @Override
        public List<OauthDTO> findAll(OauthSearchParams searchParams) {
            return BeanConvertUtils.toList(service.selectAll(searchParams));
        }
    }
}
