/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.region.web.controller.user;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.annotation.HasPermission;
import net.guerlab.smart.region.core.PermissionConstants;
import net.guerlab.smart.region.core.domain.RegionDTO;
import net.guerlab.smart.region.service.crons.DistrictSyncCron;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.web.controller.AbstractRegionController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.web.bind.annotation.*;

/**
 * 地区
 *
 * @author guer
 */
@Tag(name = "地区")
@RestController("/user/region")
@RequestMapping("/user/region")
public class RegionController extends AbstractRegionController {

    private DistrictSyncCron districtSyncCron;

    private OperationLogApi operationLogApi;

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @Operation(description = "添加", security = @SecurityRequirement(name = Constants.TOKEN))
    @PostMapping
    @CacheEvict(value = "region", allEntries = true)
    public RegionDTO save(@Parameter(name = "对象数据", required = true) @RequestBody RegionDTO dto) {
        Region entity = new Region();
        copyProperties(dto, entity);
        entity.setRegionId(dto.getRegionId());
        service.insert(entity);
        operationLogApi.add("添加地区", UserContextHandler.getUserId(), entity);
        return entity.convert();
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @Operation(description = "编辑", security = @SecurityRequirement(name = Constants.TOKEN))
    @PutMapping("/{regionId}")
    @CacheEvict(value = "region", allEntries = true)
    public RegionDTO update(@Parameter(name = "地区ID", required = true) @PathVariable Long regionId, @Parameter(name = "对象数据", required = true) @RequestBody RegionDTO dto) {
        Region entity = findOne0(regionId);
        copyProperties(dto, entity);
        service.updateById(entity);
        operationLogApi.add("编辑地区", UserContextHandler.getUserId(), entity);
        return service.selectById(regionId).convert();
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @Operation(description = "删除", security = @SecurityRequirement(name = Constants.TOKEN))
    @DeleteMapping("/{regionId}")
    @CacheEvict(value = "region", allEntries = true)
    public void delete(@Parameter(name = "地区ID", required = true) @PathVariable Long regionId, @Parameter(name = "强制删除标志") @RequestParam(required = false) Boolean force) {
        findOne0(regionId);
        service.deleteById(regionId, force);
        operationLogApi.add("删除地区", UserContextHandler.getUserId(), regionId);
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @Operation(description = "删除缓存", security = @SecurityRequirement(name = Constants.TOKEN))
    @DeleteMapping("/cache")
    @CacheEvict(value = "region", allEntries = true)
    public void deleteCache() {
        operationLogApi.add("删除地区缓存", UserContextHandler.getUserId());
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @Operation(description = "同步数据", security = @SecurityRequirement(name = Constants.TOKEN))
    @PostMapping("/sync")
    @CacheEvict(value = "region", allEntries = true)
    public void sync() {
        districtSyncCron.sync();
        operationLogApi.add("同步地区数据", UserContextHandler.getUserId());
    }

    private void copyProperties(RegionDTO dto, Region entity) {
        entity.setRegionName(dto.getRegionName());
        entity.setParentId(dto.getParentId());
        entity.setOrderNum(dto.getOrderNum());
    }

    @Autowired
    public void setDistrictSyncCron(DistrictSyncCron districtSyncCron) {
        this.districtSyncCron = districtSyncCron;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }
}
