/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.region.web.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.platform.commons.util.TreeUtils;
import net.guerlab.smart.region.core.domain.RegionDTO;
import net.guerlab.smart.region.core.exception.RegionIdInvalidException;
import net.guerlab.smart.region.core.exception.RegionInvalidException;
import net.guerlab.smart.region.core.searchparams.RegionSearchParams;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.service.RegionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;

import java.util.Collection;
import java.util.stream.Collectors;

/**
 * 抽象地区
 *
 * @author guer
 */
public class AbstractRegionController {

    protected RegionService service;

    @IgnoreLogin
    @Operation(description = "查询详情")
    @GetMapping("/{regionId}")
    public RegionDTO findOne(@Parameter(name = "地区ID", required = true) @PathVariable Long regionId) {
        return findOne0(regionId).convert();
    }

    @IgnoreLogin
    @Operation(description = "查询下级列表")
    @GetMapping("/{regionId}/children")
    @Cacheable(value = "region", key = "#regionId")
    public Collection<RegionDTO> findChildren(@Parameter(name = "regionId", required = true) @PathVariable Long regionId) {
        if (!NumberHelper.greaterZero(regionId)) {
            regionId = Constants.DEFAULT_PARENT_ID;
        }

        RegionSearchParams searchParams = new RegionSearchParams();
        searchParams.setParentId(regionId);

        return BeanConvertUtils.toList(service.selectAll(searchParams));
    }

    @IgnoreLogin
    @Operation(description = "查询深度列表")
    @GetMapping("/{regionId}/depth")
    public Collection<RegionDTO> findDepthList(@Parameter(name = "regionId", required = true) @PathVariable Long regionId) {
        return service.findDepthList(regionId).stream().map(Region::convert).collect(Collectors.toList());
    }

    @IgnoreLogin
    @Operation(description = "查询树形结构")
    @GetMapping("/tree")
    @Cacheable(value = "region", key = "'tree'")
    public Collection<RegionDTO> getTree() {
        return TreeUtils.tree(CollectionUtil.toList(service.selectAll(), Region::convert));
    }

    protected Region findOne0(Long regionId) {
        if (!NumberHelper.greaterZero(regionId)) {
            throw new RegionIdInvalidException();
        }

        return service.selectByIdOptional(regionId).orElseThrow(RegionInvalidException::new);
    }

    @Autowired
    public void setService(RegionService service) {
        this.service = service;
    }
}
