package net.guerlab.smart.region.web.controller.user;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.smart.platform.commons.annotation.HasPermission;
import net.guerlab.smart.region.core.PermissionConstants;
import net.guerlab.smart.region.core.domain.RegionDTO;
import net.guerlab.smart.region.service.crons.DistrictSyncCron;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.web.controller.AbstractRegionController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.web.bind.annotation.*;

/**
 * 地区
 *
 * @author guer
 */
@Api(tags = "地区")
@RestController("/user/region")
@RequestMapping("/user/region")
public class RegionController extends AbstractRegionController {

    private DistrictSyncCron districtSyncCron;

    private OperationLogApi operationLogApi;

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @ApiOperation("添加")
    @PostMapping
    @CacheEvict(value = "region", allEntries = true)
    public RegionDTO save(@ApiParam(value = "对象数据", required = true) @RequestBody RegionDTO dto) {
        Region entity = new Region();
        copyProperties(dto, entity);
        entity.setRegionId(dto.getRegionId());
        service.insert(entity);
        operationLogApi.add("添加地区", UserContextHandler.getUserId(), entity);
        return entity.toDTO();
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @ApiOperation("编辑")
    @PutMapping("/{regionId}")
    @CacheEvict(value = "region", allEntries = true)
    public RegionDTO update(@ApiParam(value = "地区ID", required = true) @PathVariable Long regionId,
            @ApiParam(value = "对象数据", required = true) @RequestBody RegionDTO dto) {
        Region entity = findOne0(regionId);
        copyProperties(dto, entity);
        service.updateSelectiveById(entity);
        operationLogApi.add("编辑地区", UserContextHandler.getUserId(), entity);
        return service.selectById(regionId).toDTO();
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @ApiOperation("删除")
    @DeleteMapping("/{regionId}")
    @CacheEvict(value = "region", allEntries = true)
    public void delete(@ApiParam(value = "地区ID", required = true) @PathVariable Long regionId,
            @ApiParam(value = "强制删除标志") @RequestParam(required = false) Boolean force) {
        findOne0(regionId);
        service.deleteById(regionId, force);
        operationLogApi.add("删除地区", UserContextHandler.getUserId(), regionId);
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @ApiOperation("删除缓存")
    @DeleteMapping("/cache")
    @CacheEvict(value = "region", allEntries = true)
    public void deleteCache() {
        operationLogApi.add("删除地区缓存", UserContextHandler.getUserId());
    }

    @HasPermission("hasKey(" + PermissionConstants.MANAGER + ")")
    @ApiOperation("同步数据")
    @PostMapping("/sync")
    @CacheEvict(value = "region", allEntries = true)
    public void sync() {
        districtSyncCron.sync();
        operationLogApi.add("同步地区数据", UserContextHandler.getUserId());
    }

    private void copyProperties(RegionDTO dto, Region entity) {
        entity.setRegionName(dto.getRegionName());
        entity.setParentId(dto.getParentId());
        entity.setOrderNum(dto.getOrderNum());
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }

    @Autowired
    public void setDistrictSyncCron(DistrictSyncCron districtSyncCron) {
        this.districtSyncCron = districtSyncCron;
    }
}
