/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.region.service.crons;

import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.lbs.entity.District;
import net.guerlab.smart.region.service.lbs.service.DistrictService;
import net.guerlab.smart.region.service.service.RegionService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

/**
 * 地区同步
 *
 * @author guer
 */
@Component
public class DistrictSyncCron {

    private static final int BATCH_NUMBER = 1000;

    private final ReentrantLock lock = new ReentrantLock();

    private DistrictService districtService;

    private RegionService regionService;

    @Scheduled(cron = "0 0 1 * * ? ")
    public void sync() {
        if (lock.tryLock()) {
            List<Region> all = getData(0L);

            for (int i = 0; i < all.size(); i += BATCH_NUMBER) {
                List<Region> subList = all.subList(i, Math.min(i + BATCH_NUMBER, all.size()));
                regionService.replaceBatchInsert(subList);
            }

            lock.unlock();
        }
    }

    private List<Region> getData(Long parentId) {
        List<Region> list = getRegions(parentId);
        List<Region> result = new ArrayList<>(list);

        for (Region region : list) {
            Long regionId = region.getRegionId();

            if (regionId.toString().endsWith("00")) {
                result.addAll(getData(regionId));
            }
        }

        return result;
    }

    private List<Region> getRegions(Long parentId) {
        return districtService.getChildren(parentId).stream().map(district -> toRegion(district, parentId)).collect(Collectors.toList());
    }

    private Region toRegion(District district, Long parentId) {
        String name = StringUtils.trimToNull(district.getName());
        String fullName = StringUtils.trimToNull(district.getFullName());
        Region region = new Region();
        region.setRegionId(district.getId());
        region.setRegionName(name != null ? name : fullName);
        region.setFullName(fullName);
        region.setParentId(parentId);
        region.setLatitude(district.getLocation().getLat());
        region.setLongitude(district.getLocation().getLng());
        region.setOrderNum(0);
        region.setVersion(0L);

        return region;
    }

    @Autowired
    public void setDistrictService(DistrictService districtService) {
        this.districtService = districtService;
    }

    @Autowired
    public void setRegionService(RegionService regionService) {
        this.regionService = regionService;
    }
}
