package net.guerlab.smart.region.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.exception.LatitudeInvalidException;
import net.guerlab.smart.platform.commons.exception.LongitudeInvalidException;
import net.guerlab.smart.platform.commons.util.OrderEntityUtils;
import net.guerlab.smart.platform.server.service.BaseBatchServiceImpl;
import net.guerlab.smart.region.core.exception.*;
import net.guerlab.smart.region.core.searchparams.RegionSearchParams;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.mapper.RegionMapper;
import net.guerlab.smart.region.service.service.RegionService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

/**
 * 地区服务实现
 *
 * @author guer
 */
@Service
public class RegionServiceImpl extends BaseBatchServiceImpl<Region, Long, RegionMapper> implements RegionService {

    @Override
    protected Region batchSaveBefore(Region entity) {
        if (entity == null) {
            return null;
        }

        Long regionId = entity.getRegionId();
        String name = StringUtils.trimToNull(entity.getRegionName());
        Long parentId = entity.getParentId();

        if (!NumberHelper.greaterZero(regionId) || name == null || name.length() > REGION_NAME_MAX_LENGTH
                || entity.getLongitude() == null || entity.getLatitude() == null) {
            return null;
        }

        if (!NumberHelper.greaterZero(parentId)) {
            entity.setParentId(Constants.DEFAULT_PARENT_ID);
        }

        entity.setRegionName(name);

        return entity;
    }

    @Override
    protected void insertBefore(Region entity) {
        Long regionId = entity.getRegionId();
        String name = StringUtils.trimToNull(entity.getRegionName());
        Long parentId = entity.getParentId();

        if (name == null) {
            throw new RegionNameInvalidException();
        } else if (name.length() > REGION_NAME_MAX_LENGTH) {
            throw new RegionNameLengthErrorException();
        }

        if (entity.getLongitude() == null) {
            throw new LongitudeInvalidException();
        }
        if (entity.getLatitude() == null) {
            throw new LatitudeInvalidException();
        }

        if (!NumberHelper.greaterZero(regionId)) {
            throw new RegionIdInvalidException();
        } else if (selectById(regionId) != null) {
            throw new RegionIdRepeatException();
        }

        if (NumberHelper.greaterZero(parentId)) {
            if (selectById(parentId) == null) {
                throw new ParentRegionInvalidException();
            }
        } else {
            entity.setParentId(Constants.DEFAULT_PARENT_ID);
        }

        entity.setRegionName(name);
        entity.setFullName(StringUtils.trimToEmpty(entity.getFullName()));

        OrderEntityUtils.propertiesCheck(entity);
    }

    @Override
    protected void updateBefore(Region entity) {
        String name = StringUtils.trimToNull(entity.getRegionName());

        if (name != null && name.length() > REGION_NAME_MAX_LENGTH) {
            throw new RegionNameLengthErrorException();
        }
        entity.setRegionName(name);

        Long parentId = entity.getParentId();
        if (NumberHelper.greaterZero(parentId)) {
            if (selectById(parentId) == null) {
                throw new ParentRegionInvalidException();
            }
        } else {
            entity.setParentId(null);
        }

        entity.setFullName(StringUtils.trimToNull(entity.getFullName()));
    }

    @Override
    protected void deleteBefore(Region entity, Boolean force) {
        deleteByIdBefore(entity.getRegionId(), force);
    }

    @Override
    protected void deleteByIdBefore(Long id, Boolean force) {
        RegionSearchParams searchParams = new RegionSearchParams();
        searchParams.setParentId(id);

        if (selectCount(searchParams) > 0) {
            throw new RegionDisallowDeleteException();
        }
    }
}
