package net.guerlab.smart.region.service.lbs.service.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.region.core.exception.LbsKeyInvalidException;
import net.guerlab.smart.region.core.exception.LbsQueryInvalidException;
import net.guerlab.smart.region.service.lbs.KeyLoadBalancerAgent;
import net.guerlab.smart.region.service.lbs.entity.LbsResponse;
import net.guerlab.smart.region.service.lbs.properties.LbsProperties;
import org.springframework.beans.factory.annotation.Autowired;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Arrays;
import java.util.Objects;

/**
 * 抽象服务实现
 *
 * @author guer
 */
@Slf4j
public class AbstractServiceImpl {

    protected final static Integer[] KEY_ERROR_CODES = new Integer[] { 190, 311 };

    protected final HttpClient httpClient = HttpClient.newHttpClient();

    protected KeyLoadBalancerAgent keyLoadBalancer;

    protected LbsProperties lbsProperties;

    protected ObjectMapper objectMapper;

    protected int getRetryNumber() {
        int retryNumber = lbsProperties.getRetryNumber();
        if (retryNumber <= 0) {
            retryNumber = LbsProperties.DEFAULT_RETRY_NUMBER;
        }
        return retryNumber;
    }

    protected void dataCheck(Object data) {
        if (data == null) {
            throw new LbsQueryInvalidException();
        }
    }

    protected final String getResult(String url) {
        HttpRequest request = HttpRequest.newBuilder(URI.create(url)).GET().build();

        try {
            return httpClient.send(request, HttpResponse.BodyHandlers.ofString()).body();
        } catch (Exception e) {
            log.debug(e.getLocalizedMessage(), e);
            throw new ApplicationException(e.getLocalizedMessage(), e);
        }
    }

    protected final String getKey() {
        String key = keyLoadBalancer.choose();

        if (key == null) {
            throw new LbsKeyInvalidException();
        }

        return key;
    }

    protected <T> T resultCheck(String key, LbsResponse<T> result) {
        try {
            if (result.getStatus() != 0) {
                if (Arrays.stream(KEY_ERROR_CODES)
                        .anyMatch(errorCode -> Objects.equals(errorCode, result.getStatus()))) {
                    keyLoadBalancer.markDown(key);
                }
                throw new ApplicationException(result.getMessage());
            }

            return result.getResult();
        } catch (Exception e) {
            if (e instanceof ApplicationException) {
                throw e;
            } else {
                log.debug(e.getLocalizedMessage(), e);
                throw new ApplicationException(e.getLocalizedMessage(), e);
            }
        }
    }

    @Autowired
    public void setKeyLoadBalancer(KeyLoadBalancerAgent keyLoadBalancer) {
        this.keyLoadBalancer = keyLoadBalancer;
    }

    @Autowired
    public void setLbsProperties(LbsProperties lbsProperties) {
        this.lbsProperties = lbsProperties;
    }

    @Autowired
    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }
}
