package net.guerlab.smart.region.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.util.OrderEntityUtils;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.smart.region.core.exception.*;
import net.guerlab.smart.region.core.searchparams.RegionSearchParams;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.mapper.RegionMapper;
import net.guerlab.smart.region.service.service.RegionService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

/**
 * 地区服务实现
 *
 * @author guer
 */
@Service
public class RegionServiceImpl extends BaseServiceImpl<Region, Long, RegionMapper> implements RegionService {

    @Override
    protected void insertBefore(Region entity) {
        Long regionId = entity.getRegionId();
        String name = StringUtils.trimToNull(entity.getRegionName());
        Long parentId = entity.getParentId();

        if (name == null) {
            throw new RegionNameInvalidException();
        } else if (name.length() > REGION_NAME_MAX_LENGTH) {
            throw new RegionNameLengthErrorException();
        }

        if (!NumberHelper.greaterZero(regionId)) {
            throw new RegionIdInvalidException();
        } else if (selectById(regionId) != null) {
            throw new RegionIdRepeatException();
        }

        if (NumberHelper.greaterZero(parentId)) {
            Region parent = selectById(parentId);
            if (parent == null) {
                throw new ParentRegionInvalidException();
            }
        } else {
            entity.setParentId(Constants.DEFAULT_PARENT_ID);
        }

        entity.setRegionName(name);
        OrderEntityUtils.propertiesCheck(entity);
    }

    @Override
    protected void updateBefore(Region entity) {
        String name = StringUtils.trimToNull(entity.getRegionName());

        if (name != null && name.length() > REGION_NAME_MAX_LENGTH) {
            throw new RegionNameLengthErrorException();
        }
        entity.setRegionName(name);

        Long parentId = entity.getParentId();
        if (NumberHelper.greaterZero(parentId)) {
            Region parent = selectById(parentId);
            if (parent == null) {
                throw new ParentRegionInvalidException();
            }
        } else {
            entity.setParentId(null);
        }
    }

    @Override
    protected void deleteBefore(Region entity, Boolean force) {
        deleteByIdBefore(entity.getRegionId(), force);
    }

    @Override
    protected void deleteByIdBefore(Long id, Boolean force) {
        RegionSearchParams searchParams = new RegionSearchParams();
        searchParams.setParentId(id);

        if (selectCount(searchParams) > 0) {
            throw new RegionDisallowDeleteException();
        }
    }
}
