package net.guerlab.smart.region.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.platform.commons.util.TreeUtils;
import net.guerlab.smart.region.api.RegionApi;
import net.guerlab.smart.region.core.domain.RegionDTO;
import net.guerlab.smart.region.core.exception.RegionInvalidException;
import net.guerlab.smart.region.core.searchparams.RegionSearchParams;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.service.RegionService;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.lang.NonNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author guer
 */
@Configuration
@Conditional(RegionApiLocalServiceAutoConfigure.WrapperCondition.class)
public class RegionApiLocalServiceAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnBean(RegionService.class)
    public RegionApi regionApiLocalServiceWrapper(RegionService service) {
        return new RegionApiLocalServiceWrapper(service);
    }

    @SuppressWarnings("WeakerAccess")
    static class WrapperCondition implements Condition {

        @Override
        public boolean matches(@NonNull ConditionContext context, @NonNull AnnotatedTypeMetadata metadata) {
            try {
                return WrapperCondition.class.getClassLoader()
                        .loadClass("net.guerlab.smart.region.service.service.RegionService") != null;
            } catch (Exception e) {
                return false;
            }
        }
    }

    @AllArgsConstructor
    private static class RegionApiLocalServiceWrapper implements RegionApi {

        private final RegionService service;

        @Override
        public RegionDTO findOne(Long areaId) {
            return service.selectByIdOptional(areaId).orElseThrow(RegionInvalidException::new).toDTO();
        }

        @Override
        public List<RegionDTO> findChildren(Long regionId) {
            if (!NumberHelper.greaterZero(regionId)) {
                return Collections.emptyList();
            }

            RegionSearchParams searchParams = new RegionSearchParams();
            searchParams.setParentId(regionId);

            return BeanConvertUtils.toList(service.selectAll(searchParams));
        }

        @Override
        public List<RegionDTO> findDepthList(Long regionId) {
            return new ArrayList<>(CollectionUtil.toList(service.findDepthList(regionId), Region::toDTO));
        }

        @Override
        public List<RegionDTO> tree() {
            return new ArrayList<>(TreeUtils.tree(CollectionUtil.toList(service.selectAll(), Region::toDTO)));
        }
    }

}
