package net.guerlab.smart.region.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.smart.platform.commons.util.TreeUtils;
import net.guerlab.smart.region.api.RegionApi;
import net.guerlab.smart.region.core.domain.RegionDTO;
import net.guerlab.smart.region.core.exception.RegionInvalidException;
import net.guerlab.smart.region.core.searchparams.RegionSearchParams;
import net.guerlab.smart.region.service.entity.Region;
import net.guerlab.smart.region.service.service.RegionService;
import net.guerlab.web.result.ListObject;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.lang.NonNull;

import java.util.ArrayList;
import java.util.List;

/**
 * @author guer
 */
@Configuration
@Conditional(RegionApiLocalServiceAutoConfigure.WrapperCondition.class)
public class RegionApiLocalServiceAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnBean(RegionService.class)
    public RegionApi regionApiLocalServiceWrapper(RegionService service) {
        return new RegionApiLocalServiceWrapper(service);
    }

    @SuppressWarnings("WeakerAccess")
    static class WrapperCondition implements Condition {

        @Override
        public boolean matches(@NonNull ConditionContext context, @NonNull AnnotatedTypeMetadata metadata) {
            try {
                return WrapperCondition.class.getClassLoader()
                        .loadClass("net.guerlab.smart.region.service.service.RegionService") != null;
            } catch (Exception e) {
                return false;
            }
        }
    }

    @AllArgsConstructor
    private static class RegionApiLocalServiceWrapper implements RegionApi {

        private RegionService service;

        @Override
        public RegionDTO findOne(Long areaId) {
            return service.selectByIdOptional(areaId).orElseThrow(RegionInvalidException::new).toDTO();
        }

        @Override
        public ListObject<RegionDTO> findList(RegionSearchParams searchParams) {
            return BeanConvertUtils.toListObject(service.selectPage(searchParams));
        }

        @Override
        public List<RegionDTO> findAll(RegionSearchParams searchParams) {
            return BeanConvertUtils.toList(service.selectAll(searchParams));
        }

        @Override
        public List<RegionDTO> tree() {
            return new ArrayList<>(TreeUtils.tree(CollectionUtil.toList(service.selectAll(), Region::toDTO)));
        }

        @Override
        public List<RegionDTO> findDepthList(Long regionId) {
            return new ArrayList<>(CollectionUtil.toList(service.findDepthList(regionId), Region::toDTO));
        }
    }

}
