/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.platform.basic.gateway.vc;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.loadbalancer.*;
import org.springframework.cloud.loadbalancer.core.NoopServiceInstanceListSupplier;
import org.springframework.cloud.loadbalancer.core.ReactorServiceInstanceLoadBalancer;
import org.springframework.cloud.loadbalancer.core.SelectedInstanceCallback;
import org.springframework.cloud.loadbalancer.core.ServiceInstanceListSupplier;
import reactor.core.publisher.Mono;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 带版本控制的服务实例负载均衡器实现
 *
 * @author guer
 */
@Slf4j
public class VersionControlReactorServiceInstanceLoadBalancer implements ReactorServiceInstanceLoadBalancer {

    private final String serviceId;

    private final ObjectProvider<ServiceInstanceListSupplier> serviceInstanceListSupplierProvider;

    private final VersionControlProperties versionControlProperties;

    private final AtomicInteger position;

    public VersionControlReactorServiceInstanceLoadBalancer(
            ObjectProvider<ServiceInstanceListSupplier> serviceInstanceListSupplierProvider, String serviceId,
            VersionControlProperties versionControlProperties) {
        this.serviceId = serviceId;
        this.serviceInstanceListSupplierProvider = serviceInstanceListSupplierProvider;
        this.versionControlProperties = versionControlProperties;
        this.position = new AtomicInteger(new Random().nextInt(1000));
    }

    @SuppressWarnings("unchecked")
    @Override
    public Mono<Response<ServiceInstance>> choose(Request request) {
        if (!versionControlProperties.isEnable()) {
            return getDefault(request);
        }

        String requestKey = StringUtils.trimToNull(versionControlProperties.getRequestKey());
        String metadataKey = StringUtils.trimToNull(versionControlProperties.getMetadataKey());
        if (requestKey == null || metadataKey == null) {
            return getDefault(request);
        }

        Request<RequestDataContext> lbRequest;
        try {
            lbRequest = (Request<RequestDataContext>) request;
        } catch (Exception e) {
            return getDefault(request);
        }

        String requestVersion = lbRequest.getContext().getClientRequest().getHeaders().getFirst(requestKey);
        if (requestVersion == null) {
            return getDefault(request);
        }

        try {
            ServiceInstanceListSupplier supplier = serviceInstanceListSupplierProvider
                    .getIfAvailable(NoopServiceInstanceListSupplier::new);
            return supplier.get(request).next().map(serviceInstances -> {
                Response<ServiceInstance> serviceInstanceResponse = getInstanceResponseWithRequestVersion(
                        serviceInstances, requestVersion, metadataKey);
                if (supplier instanceof SelectedInstanceCallback && serviceInstanceResponse.hasServer()) {
                    ((SelectedInstanceCallback) supplier).selectedServiceInstance(serviceInstanceResponse.getServer());
                }
                return serviceInstanceResponse;
            });
        } catch (Exception e) {
            log.debug(e.getLocalizedMessage(), e);
            throw e;
        }
    }

    private Response<ServiceInstance> getInstanceResponseWithRequestVersion(List<ServiceInstance> instances,
            String requestVersion, String metadataKey) {
        if (instances.isEmpty()) {
            log.warn("No servers available for service: {}", serviceId);
            return new EmptyResponse();
        }

        // 空列表
        List<ServiceInstance> hasNotVersions = new ArrayList<>(instances.size());
        // 相等列表
        List<ServiceInstance> equalsVersions = new ArrayList<>(instances.size());
        // 可匹配列表
        List<ServiceInstance> matchVersions = new ArrayList<>(instances.size());

        // 按照解析出的版本号信息进行分组
        instances.forEach(instance -> {
            String version = instance.getMetadata().get(metadataKey);
            if (version == null) {
                hasNotVersions.add(instance);
            } else if (version.equals(requestVersion)) {
                equalsVersions.add(instance);
            } else if (VersionCompareUtils.match(requestVersion, version)) {
                matchVersions.add(instance);
            }
            // 其他情况进行忽略
        });

        if (!equalsVersions.isEmpty()) {
            return getInstanceResponse(equalsVersions);
        } else if (!matchVersions.isEmpty()) {
            return getInstanceResponse(matchVersions);
        } else if (!hasNotVersions.isEmpty()) {
            return getInstanceResponse(hasNotVersions);
        } else {
            return new EmptyResponse();
        }
    }

    private Mono<Response<ServiceInstance>> getDefault(Request<?> request) {
        ServiceInstanceListSupplier supplier = serviceInstanceListSupplierProvider
                .getIfAvailable(NoopServiceInstanceListSupplier::new);
        return supplier.get(request).next()
                .map(serviceInstances -> processInstanceResponse(supplier, serviceInstances));
    }

    private Response<ServiceInstance> processInstanceResponse(ServiceInstanceListSupplier supplier,
            List<ServiceInstance> serviceInstances) {
        Response<ServiceInstance> serviceInstanceResponse = getInstanceResponse(serviceInstances);
        if (supplier instanceof SelectedInstanceCallback && serviceInstanceResponse.hasServer()) {
            ((SelectedInstanceCallback) supplier).selectedServiceInstance(serviceInstanceResponse.getServer());
        }
        return serviceInstanceResponse;
    }

    private Response<ServiceInstance> getInstanceResponse(List<ServiceInstance> instances) {
        if (instances.isEmpty()) {
            log.warn("No servers available for service: " + serviceId);
            return new EmptyResponse();
        }
        int pos = Math.abs(this.position.incrementAndGet());
        ServiceInstance instance = instances.get(pos % instances.size());
        return new DefaultResponse(instance);
    }
}
