/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.platform.basic.gateway.protect;

import lombok.extern.slf4j.Slf4j;
import net.guerlab.commons.collection.CollectionUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.gateway.filter.GatewayFilter;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.core.Ordered;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;
import org.springframework.web.server.ResponseStatusException;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import java.util.List;
import java.util.Objects;

/**
 * 接口保护请求过滤器
 *
 * @author guer
 */
@Slf4j
public class ProtectFilter implements GatewayFilter, GlobalFilter, Ordered {

    private final ProtectProperties properties;

    private final AntPathMatcher matcher = new AntPathMatcher();

    public ProtectFilter(ProtectProperties properties) {
        this.properties = properties;
    }

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        if (!properties.isEnable()) {
            return chain.filter(exchange);
        }

        List<ProtectProperties.Url> urls = properties.getUrls();

        if (CollectionUtil.isEmpty(urls)) {
            return chain.filter(exchange);
        }

        boolean release = false;
        String releaseName = StringUtils.trimToNull(properties.getReleaseHeaderName());
        String releaseValue = StringUtils.trimToNull(properties.getReleaseHeaderValue());
        String requestValue = null;
        if (releaseName != null && releaseValue != null) {
            requestValue = exchange.getRequest().getHeaders().getFirst(releaseName);
            release = requestValue != null && Objects.equals(requestValue, releaseValue);
        }

        HttpMethod requestMethod = exchange.getRequest().getMethod();
        String requestPath = exchange.getRequest().getURI().getPath();

        for (ProtectProperties.Url url : urls) {
            String path = StringUtils.trimToNull(url.getPath());
            if (path == null) {
                continue;
            }

            HttpMethod method = url.getMethod();
            if (method != null && Objects.equals(requestMethod, method)) {
                continue;
            }

            if (matcher.match(path, requestPath)) {
                log.debug("intercept path info: [url: {}, path: {}]]", path, requestPath);
                log.debug("release info: [name: {}, value: {}, request: {}, release: {}]", releaseName, releaseValue, requestValue, release);
                if (!release) {
                    return createNotFoundError();
                }
            }
        }

        return chain.filter(exchange);
    }

    private <R> Mono<R> createNotFoundError() {
        return Mono.defer(() -> {
            Exception ex = new ResponseStatusException(HttpStatus.NOT_FOUND, "No matching handler");
            return Mono.error(ex);
        });
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }
}
