/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.platform.basic.gateway.logger;

import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

/**
 * 请求时间记录
 *
 * @author guer
 */
@Slf4j
@Order(Ordered.HIGHEST_PRECEDENCE)
public class RequestTimeFilter implements GlobalFilter {

    private static final String REQUEST_TIME = "requestTimeStartWith";

    private final LoggerProperties properties;

    public RequestTimeFilter(LoggerProperties properties) {
        this.properties = properties;
    }

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        if (!properties.isEnableRecordRequestTime()) {
            return chain.filter(exchange);
        }

        exchange.getAttributes().put(REQUEST_TIME, System.currentTimeMillis());
        return chain.filter(exchange).then(Mono.fromCallable(() -> {
            Long startTime = exchange.getAttribute(REQUEST_TIME);
            if (startTime != null) {
                log.info("request [{}]:[{}] used: {} milliseconds", exchange.getRequest().getMethod(),
                        exchange.getRequest().getPath(), System.currentTimeMillis() - startTime);
            }
            return null;
        }));
    }
}

