package net.guerlab.smart.pay.web.controller.user;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.pay.core.domain.WxPayConfigDTO;
import net.guerlab.smart.pay.core.exception.WxPayConfigInvalidException;
import net.guerlab.smart.pay.core.searchparams.WxPayConfigSearchParams;
import net.guerlab.smart.pay.service.entity.WxPayConfig;
import net.guerlab.smart.pay.service.service.WxPayConfigService;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.platform.server.controller.BaseController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import net.guerlab.web.result.Fail;
import net.guerlab.web.result.Result;
import net.guerlab.web.result.Succeed;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.util.Base64;

/**
 * 微信支付配置
 *
 * @author guer
 */
@Api(tags = "微信支付配置")
@RestController("/user/wxPayConfig")
@RequestMapping("/user/wxPayConfig")
public class WxPayConfigController
        extends BaseController<WxPayConfigDTO, WxPayConfig, WxPayConfigService, WxPayConfigSearchParams, String> {

    private OperationLogApi operationLogApi;

    @IgnoreLogin
    @ApiOperation("解析证书文件base64")
    @PostMapping("/parseCertFile")
    public Result<String> parseCertFile(MultipartFile file) {
        try {
            return new Succeed<>(Succeed.MSG, Base64.getEncoder().encodeToString(file.getBytes()));
        } catch (Exception e) {
            return new Fail<>(e.getMessage());
        }
    }

    @ApiOperation("启用配置")
    @PostMapping("/{id}/enable")
    public WxPayConfigDTO enable(@ApiParam(value = "id", required = true) @PathVariable String id) {
        WxPayConfig entity = findOne0(id);
        entity.setEnable(true);
        getService().updateSelectiveById(entity);
        operationLogApi.add("启用微信支付配置", UserContextHandler.getUserId(), id);
        return entity.toDTO();
    }

    @ApiOperation("禁用配置")
    @PostMapping("/{id}/disable")
    public WxPayConfigDTO disable(@ApiParam(value = "id", required = true) @PathVariable String id) {
        WxPayConfig entity = findOne0(id);
        entity.setEnable(false);
        getService().updateSelectiveById(entity);
        operationLogApi.add("禁用微信支付配置", UserContextHandler.getUserId(), id);
        return entity.toDTO();
    }

    @Override
    protected ApplicationException nullPointException() {
        return new WxPayConfigInvalidException();
    }

    @Override
    public void copyProperties(WxPayConfigDTO dto, WxPayConfig entity, String id) {
        super.copyProperties(dto, entity, id);
        if (id != null) {
            entity.setAppId(id);
        }
    }

    @Override
    public void afterSave(WxPayConfig entity, WxPayConfigDTO dto) {
        operationLogApi.add("新增微信支付配置", UserContextHandler.getUserId(), entity);
    }

    @Override
    public void afterUpdate(WxPayConfig entity, WxPayConfigDTO dto) {
        operationLogApi.add("编辑微信支付配置", UserContextHandler.getUserId(), entity);
    }

    @Override
    public void afterDelete(WxPayConfig entity) {
        operationLogApi.add("删除微信支付配置", UserContextHandler.getUserId(), entity.getAppId());
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }
}
