package net.guerlab.smart.pay.web.controller.user;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.pay.core.domain.PayLogNotifyDTO;
import net.guerlab.smart.pay.core.exception.ExceptionReasonInvalidException;
import net.guerlab.smart.pay.core.exception.ExceptionReasonLengthErrorException;
import net.guerlab.smart.pay.core.exception.PayLogNotifyInvalidException;
import net.guerlab.smart.pay.core.searchparams.PayLogNotifySearchParams;
import net.guerlab.smart.pay.service.entity.PayLogNotify;
import net.guerlab.smart.pay.service.service.PayLogNotifyService;
import net.guerlab.smart.pay.web.domain.ExceptionDTO;
import net.guerlab.smart.pay.web.excel.PayLogNotifyExcelExport;
import net.guerlab.smart.platform.excel.ExcelUtils;
import net.guerlab.smart.platform.server.controller.BaseFindController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;

/**
 * 支付记录通知
 *
 * @author guer
 */
@Api(tags = "支付记录通知")
@RestController("/user/payLogNotify")
@RequestMapping("/user/payLogNotify")
public class PayLogNotifyController
        extends BaseFindController<PayLogNotifyDTO, PayLogNotify, PayLogNotifyService, PayLogNotifySearchParams, Long> {

    private OperationLogApi operationLogApi;

    @ApiOperation("导出Excel")
    @GetMapping("/exportExcel")
    public void exportExcel(HttpServletResponse response, PayLogNotifySearchParams searchParams) {
        beforeFind(searchParams);
        ExcelUtils.exportExcel(response, getService().selectAll(searchParams), PayLogNotifyExcelExport.class,
                "PayLogNotify-" + System.currentTimeMillis());
        operationLogApi.add("导出支付记录通知列表", UserContextHandler.getUserId(), searchParams);
    }

    @ApiOperation("标记异常")
    @PostMapping("/{id}/markException")
    @Transactional(rollbackFor = Exception.class)
    public PayLogNotifyDTO markException(@ApiParam(value = "id", required = true) @PathVariable Long id,
            @RequestBody ExceptionDTO dto) {
        String reason = StringUtils.trimToNull(dto.getReason());

        if (reason == null) {
            throw new ExceptionReasonInvalidException();
        } else if (reason.length() > ExceptionDTO.REASON_MAX_LENGTH) {
            throw new ExceptionReasonLengthErrorException();
        }

        findOne0(id);
        getService().markException(id, reason);
        operationLogApi.add("标记异常-支付记录通知", UserContextHandler.getUserId(), id, reason);

        return findOne0(id).toDTO();
    }

    @ApiOperation("移除标记异常")
    @PostMapping("/{id}/removeMarkException")
    @Transactional(rollbackFor = Exception.class)
    public PayLogNotifyDTO removeMarkException(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        findOne0(id);
        getService().removeExceptionMark(id);
        operationLogApi.add("移除异常标记-支付记录通知", UserContextHandler.getUserId(), id);

        return findOne0(id).toDTO();
    }

    @Override
    protected ApplicationException nullPointException() {
        return new PayLogNotifyInvalidException();
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }
}
