package net.guerlab.smart.pay.web.controller.user;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.pay.core.domain.PayLogExtends;
import net.guerlab.smart.pay.core.domain.PayOrderDTO;
import net.guerlab.smart.pay.core.enums.PayStatus;
import net.guerlab.smart.pay.core.exception.ExceptionReasonInvalidException;
import net.guerlab.smart.pay.core.exception.ExceptionReasonLengthErrorException;
import net.guerlab.smart.pay.core.exception.PayOrderInvalidException;
import net.guerlab.smart.pay.core.exception.PayStatusErrorException;
import net.guerlab.smart.pay.core.searchparams.PayOrderSearchParams;
import net.guerlab.smart.pay.service.entity.PayLog;
import net.guerlab.smart.pay.service.entity.PayOrder;
import net.guerlab.smart.pay.service.service.PayLogService;
import net.guerlab.smart.pay.service.service.PayOrderService;
import net.guerlab.smart.pay.web.domain.ExceptionDTO;
import net.guerlab.smart.pay.web.excel.PayOrderExcelExport;
import net.guerlab.smart.platform.excel.ExcelUtils;
import net.guerlab.smart.platform.server.controller.BaseFindController;
import net.guerlab.smart.platform.user.api.OperationLogApi;
import net.guerlab.smart.platform.user.auth.UserContextHandler;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 支付订单
 *
 * @author guer
 */
@Api(tags = "支付订单")
@RestController("/user/payOrder")
@RequestMapping("/user/payOrder")
public class PayOrderController
        extends BaseFindController<PayOrderDTO, PayOrder, PayOrderService, PayOrderSearchParams, Long> {

    private PayLogService logService;

    private OperationLogApi operationLogApi;

    @Override
    protected ApplicationException nullPointException() {
        return new PayOrderInvalidException();
    }

    @ApiOperation("导出Excel")
    @GetMapping("/exportExcel")
    public void exportExcel(HttpServletResponse response, PayOrderSearchParams searchParams) throws IOException {
        beforeFind(searchParams);
        ExcelUtils.exportExcel(response, getService().selectAll(searchParams), PayOrderExcelExport.class,
                "PayOrder-" + System.currentTimeMillis());
        operationLogApi.add("导出支付订单列表", UserContextHandler.getUserId(), searchParams);
    }

    @ApiOperation("确认支付")
    @PostMapping("/{id}/confirmPayed")
    @Transactional(rollbackFor = Exception.class)
    public PayOrderDTO confirmPayed(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        PayOrder payOrder = findOne0(id);

        if (payOrder.getPayStatus() != PayStatus.WAIT_PAY) {
            throw new PayStatusErrorException();
        }

        PayLogExtends payLogExtends = new PayLogExtends();
        payLogExtends.put("operation_userId", UserContextHandler.getUserIdString());
        payLogExtends.put("operation_name", UserContextHandler.getName());
        payLogExtends.put("operation_departmentId", UserContextHandler.getDepartmentIdString());
        payLogExtends.put("operation_departmentName", UserContextHandler.getDepartmentName());

        PayLog payLog = logService.create(id, "MANUAL_CONFIRM", payLogExtends);
        payLog.setVersion(1L);
        logService.payed(payLog);

        operationLogApi.add("确认支付", UserContextHandler.getUserId(), payOrder);

        return findOne0(id).toDTO();
    }

    @ApiOperation("标记异常")
    @PostMapping("/{id}/markException")
    @Transactional(rollbackFor = Exception.class)
    public PayOrderDTO markException(@ApiParam(value = "id", required = true) @PathVariable Long id,
            @RequestBody ExceptionDTO dto) {
        String reason = StringUtils.trimToNull(dto.getReason());

        if (reason == null) {
            throw new ExceptionReasonInvalidException();
        } else if (reason.length() > ExceptionDTO.REASON_MAX_LENGTH) {
            throw new ExceptionReasonLengthErrorException();
        }

        findOne0(id);
        getService().markException(id, reason);
        operationLogApi.add("标记异常-支付订单", UserContextHandler.getUserId(), id, reason);

        return findOne0(id).toDTO();
    }

    @ApiOperation("移除标记异常")
    @PostMapping("/{id}/removeMarkException")
    @Transactional(rollbackFor = Exception.class)
    public PayOrderDTO removeMarkException(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        findOne0(id);
        getService().removeExceptionMark(id);
        operationLogApi.add("移除异常标记-支付订单", UserContextHandler.getUserId(), id);

        return findOne0(id).toDTO();
    }

    @Autowired
    public void setLogService(PayLogService logService) {
        this.logService = logService;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }
}
