package net.guerlab.smart.pay.web.controller;

import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import net.guerlab.smart.pay.service.entity.PayLog;
import net.guerlab.smart.pay.service.service.PayLogService;
import net.guerlab.smart.pay.service.wx.WxPayServiceConstant;
import net.guerlab.smart.platform.server.utils.IpUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import java.time.format.DateTimeFormatter;
import java.util.Objects;

/**
 * 抽象微信支付请求控制器
 *
 * @author guer
 */
@Slf4j
public abstract class AbstractWxPayRequestController extends AbstractWxPayController {

    /**
     * 支付记录服务
     */
    private PayLogService logService;

    @ApiOperation("通过支付订单ID请求支付")
    @GetMapping("/byPayOrderId/{appId}/{payOrderId}")
    public final Object byPayOrderId(@ApiParam(value = "openId", required = true) @RequestParam("openId") String openId,
            @ApiParam(value = "微信应用ID", required = true) @PathVariable String appId,
            @ApiParam(value = "支付订单ID", required = true) @PathVariable Long payOrderId, HttpServletRequest request) {
        PayLog payLog = logService.create(payOrderId, getPayChannel(), appId);

        return pay0(openId, appId, payLog, request);
    }

    @ApiOperation("通过业务分组和业务ID请求支付")
    @GetMapping("/byBusinessId/{appId}/{businessGroup}/{businessId}")
    public final Object byBusinessId(@ApiParam(value = "openId", required = true) @RequestParam("openId") String openId,
            @ApiParam(value = "微信应用ID", required = true) @PathVariable String appId,
            @ApiParam(value = "业务分组", required = true) @PathVariable String businessGroup,
            @ApiParam(value = "业务ID", required = true) @PathVariable Long businessId, HttpServletRequest request) {
        PayLog payLog = logService.create(businessGroup, businessId, getPayChannel(), appId);

        return pay0(openId, appId, payLog, request);
    }

    private <T> T pay0(String openId, String appId, PayLog payLog, HttpServletRequest request) {
        WxPayUnifiedOrderRequest orderRequest = new WxPayUnifiedOrderRequest();
        orderRequest.setOpenid(openId);
        orderRequest.setAppid(appId);
        orderRequest.setBody(payLog.getOrderTitle());
        orderRequest.setOutTradeNo(Objects.toString(payLog.getPayLogId()));
        orderRequest.setTotalFee(payLog.getAmount().multiply(WxPayServiceConstant.HUNDRED).intValue());
        orderRequest.setSpbillCreateIp(IpUtils.getIp(request));
        orderRequest.setTimeExpire(payLog.getPayTimeoutTime().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")));
        orderRequest.setNotifyUrl(getNotifyUrl());

        return wxPayService.createOrder(orderRequest);
    }

    /**
     * 获取通知URL
     *
     * @return 通知URL
     */
    protected abstract String getNotifyUrl();

    @Autowired
    public void setLogService(PayLogService logService) {
        this.logService = logService;
    }
}
