package net.guerlab.smart.license.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.license.core.domain.LicenseExtends;
import net.guerlab.smart.license.core.exception.*;
import net.guerlab.smart.license.core.searchparams.LicenseSearchParams;
import net.guerlab.smart.license.service.entity.License;
import net.guerlab.smart.license.service.entity.LicenseGroup;
import net.guerlab.smart.license.service.mapper.LicenseMapper;
import net.guerlab.smart.license.service.service.AfterLicenseGroupUpdateHandler;
import net.guerlab.smart.license.service.service.LicenseGroupService;
import net.guerlab.smart.license.service.service.LicenseService;
import net.guerlab.smart.license.service.utils.LicenseHelper;
import net.guerlab.smart.license.service.utils.RsaKey;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Optional;

/**
 * 授权服务实现
 *
 * @author guer
 */
@Service
public class LicenseServiceImpl extends BaseServiceImpl<License, Long, LicenseMapper>
        implements LicenseService, AfterLicenseGroupUpdateHandler {

    private LicenseGroupService groupService;

    @Override
    public void setEffective(LocalDate date) {
        if (date != null) {
            mapper.setEffective(date);
        }
    }

    @Override
    public void setInvalid(LocalDate date) {
        if (date != null) {
            mapper.setInvalid(date);
        }
    }

    @Override
    public void revoked(Long licenseId) {
        if (NumberHelper.greaterZero(licenseId)) {
            mapper.revoked(licenseId);
        }
    }

    private void setLicenseCode(License entity, LicenseGroup licenseGroup) {
        String publicKey = licenseGroup.getPublicKey();
        String licenseCodeData = LicenseHelper.createLicenseCodeData(entity,
                new RsaKey().setPublicKey(publicKey).setPrivateKey(licenseGroup.getPrivateKey()));
        List<Object> params = Arrays.asList(entity.getLicenseId(), entity.getLicenseTo(), entity.getLicenseStartDate(),
                entity.getLicenseEndDate(), licenseCodeData, publicKey);
        String licenseCode = StringUtils.join(params, LicenseHelper.LICENSE_FILE_CONTENT_SEPARATOR);
        entity.setLicenseCode(Base64.getEncoder().encodeToString(licenseCode.getBytes()));
        entity.setPublicKey(publicKey);
    }

    @Override
    public void afterLicenseGroupUpdateHandler(LicenseGroup licenseGroup) {
        if (licenseGroup == null) {
            return;
        }

        Long licenseGroupId = licenseGroup.getLicenseGroupId();
        String licenseGroupName = StringUtils.trimToNull(licenseGroup.getLicenseGroupName());

        if (!NumberHelper.greaterZero(licenseGroupId) || licenseGroupName == null) {
            return;
        }

        LicenseSearchParams searchParams = new LicenseSearchParams();
        searchParams.setLicenseGroupId(licenseGroupId);

        License license = new License();
        license.setLicenseGroupName(licenseGroupName);

        mapper.updateByExampleSelective(license, getExample(searchParams));
    }

    private LicenseGroup findLicenseGroup(Long licenseGroupId) {
        if (!NumberHelper.greaterZero(licenseGroupId)) {
            throw new LicenseGroupIdInvalidException();
        }

        return groupService.selectByIdOptional(licenseGroupId).orElseThrow(LicenseGroupInvalidException::new);
    }

    @Override
    protected void insertBefore(License entity) {
        String licenseTo = StringUtils.trimToNull(entity.getLicenseTo());
        Long licenseGroupId = entity.getLicenseGroupId();
        LocalDate startDate = entity.getLicenseStartDate();
        LocalDate endDate = entity.getLicenseEndDate();

        if (licenseTo == null || licenseTo.contains(LicenseHelper.LICENSE_FILE_CONTENT_SEPARATOR)) {
            throw new LicenseToInvalidException();
        } else if (licenseTo.length() > LICENSE_TO_MAX_LENGTH) {
            throw new LicenseToLengthErrorException();
        }
        if (startDate == null) {
            throw new LicenseStartDateInvalidException();
        }
        if (endDate == null || startDate.isAfter(endDate)) {
            throw new LicenseEndDateInvalidException();
        }

        LicenseGroup licenseGroup = findLicenseGroup(licenseGroupId);

        entity.setLicenseId(sequence.nextId());
        entity.setLicenseTo(licenseTo);
        entity.setLicenseGroupName(licenseGroup.getLicenseGroupName());
        entity.setRemark(StringUtils.trimToEmpty(entity.getRemark()));
        entity.setCreateTime(LocalDateTime.now());
        entity.setLicenseStartDate(startDate);
        entity.setLicenseEndDate(endDate);
        entity.setRevoked(false);

        LocalDate now = LocalDate.now();
        entity.setEffective(!now.isBefore(startDate) && !endDate.isBefore(now));

        saveLicenseExtends(entity, Optional.ofNullable(entity.getLicenseExtends()).orElseGet(LicenseExtends::new));
        setLicenseCode(entity, licenseGroup);
    }

    @Override
    protected void insertAfter(License entity) {
        groupService.addLicensedNumber(entity.getLicenseGroupId());
    }

    @Override
    protected void updateBefore(License entity) {
        entity.setRemark(StringUtils.trimToNull(entity.getRemark()));
        entity.setLicenseGroupName(null);
    }

    private void saveLicenseExtends(License entity, LicenseExtends licenseExtends) {
        LicenseExtends saveLicenseExtends = new LicenseExtends();
        licenseExtends.forEach((key, value) -> {
            String k = StringUtils.trimToNull(key);
            String v = StringUtils.trimToNull(value);
            if (k != null && v != null) {
                saveLicenseExtends.put(k, v);
            }
        });

        entity.setLicenseExtends(saveLicenseExtends);
    }

    @Autowired
    public void setGroupService(LicenseGroupService groupService) {
        this.groupService = groupService;
    }
}
