package net.guerlab.smart.license.service.utils;

import io.jsonwebtoken.*;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.license.core.exception.LicenseBeOverdueException;
import net.guerlab.smart.license.core.exception.LicenseParseFailException;
import net.guerlab.smart.license.service.entity.License;

import java.security.Key;
import java.security.KeyFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;

/**
 * 授权码工具类
 *
 * @author guer
 */
public class LicenseCodeHelper {

    public static final String ALGORITHM = "RSA";

    private LicenseCodeHelper() {
    }

    /**
     * 创建授权码内容
     *
     * @param license
     *         授权对象
     * @param rsaKey
     *         公私钥
     * @return 授权码内容
     */
    public static String createLicenseCodeData(License license, RsaKey rsaKey) {
        JwtBuilder builder = Jwts.builder();
        builder.setHeaderParam("typ", "JWT");
        builder.setSubject(license.getLicenseTo());
        builder.setNotBefore(toDate(license.getLicenseStartDate()));
        builder.setExpiration(toDate(license.getLicenseEndDate()));
        builder.claim("licenseId", license.getLicenseId());

        try {
            Key key = KeyFactory.getInstance(ALGORITHM)
                    .generatePrivate(new PKCS8EncodedKeySpec(rsaKey.getPrivateKey()));

            builder.signWith(SignatureAlgorithm.RS512, key);

            return builder.compact();
        } catch (Exception e) {
            throw new ApplicationException(e.getLocalizedMessage(), e);
        }
    }

    private static Date toDate(LocalDate date) {
        return Date.from(date.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    public static Claims parse(String licenseCode, RsaKey rsaKey) {
        try {
            Key key = KeyFactory.getInstance(ALGORITHM).generatePublic(new X509EncodedKeySpec(rsaKey.getPublicKey()));

            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(key).parseClaimsJws(licenseCode);

            return claimsJws.getBody();
        } catch (ExpiredJwtException e) {
            throw new LicenseBeOverdueException();
        } catch (MalformedJwtException | SignatureException | UnsupportedJwtException | IllegalArgumentException e) {
            throw new LicenseParseFailException();
        } catch (Exception e) {
            throw new ApplicationException(e.getLocalizedMessage(), e);
        }
    }
}
