package net.guerlab.smart.license.service.service.impl;

import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.license.core.exception.LicenseGroupCannotDeleteException;
import net.guerlab.smart.license.core.exception.LicenseGroupIdInvalidException;
import net.guerlab.smart.license.core.exception.LicenseGroupNameInvalidException;
import net.guerlab.smart.license.core.searchparams.LicenseSearchParams;
import net.guerlab.smart.license.service.entity.LicenseGroup;
import net.guerlab.smart.license.service.mapper.LicenseGroupMapper;
import net.guerlab.smart.license.service.service.AfterLicenseGroupUpdateHandler;
import net.guerlab.smart.license.service.service.LicenseGroupService;
import net.guerlab.smart.license.service.service.LicenseService;
import net.guerlab.smart.license.service.utils.LicenseCodeHelper;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.spring.commons.util.SpringApplicationContextUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.time.LocalDateTime;
import java.util.Base64;

/**
 * 授权组服务实现
 *
 * @author guer
 */
@Service
public class LicenseGroupServiceImpl extends BaseServiceImpl<LicenseGroup, Long, LicenseGroupMapper>
        implements LicenseGroupService {

    private LicenseService licenseService;

    @Override
    public void addLicensedNumber(Long licenseGroupId) {
        if (!NumberHelper.greaterZero(licenseGroupId)) {
            throw new LicenseGroupIdInvalidException();
        }

        mapper.addLicensedNumber(licenseGroupId);
    }

    @Override
    public void reduceLicensedNumber(Long licenseGroupId) {
        if (!NumberHelper.greaterZero(licenseGroupId)) {
            throw new LicenseGroupIdInvalidException();
        }

        mapper.reduceLicensedNumber(licenseGroupId);
    }

    @Override
    protected void insertBefore(LicenseGroup entity) {
        String licenseGroupName = StringUtils.trimToNull(entity.getLicenseGroupName());

        if (licenseGroupName == null) {
            throw new LicenseGroupNameInvalidException();
        }

        entity.setLicenseGroupId(sequence.nextId());
        entity.setLicenseGroupName(licenseGroupName);
        entity.setRemark(StringUtils.trimToEmpty(entity.getRemark()));
        entity.setCreateTime(LocalDateTime.now());
        entity.setLicensedNumber(0L);

        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(LicenseCodeHelper.ALGORITHM);
            keyPairGenerator.initialize(2048);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();
            entity.setPublicKey(Base64.getEncoder().encodeToString(keyPair.getPublic().getEncoded()));
            entity.setPrivateKey(Base64.getEncoder().encodeToString(keyPair.getPrivate().getEncoded()));
        } catch (Exception e) {
            throw new ApplicationException(e.getLocalizedMessage(), e);
        }
    }

    @Override
    protected void updateBefore(LicenseGroup entity) {
        entity.setLicensedNumber(null);
    }

    @Override
    protected void updateAfter(LicenseGroup entity) {
        SpringApplicationContextUtil.getContext().getBeansOfType(AfterLicenseGroupUpdateHandler.class).values()
                .forEach(handler -> handler.afterLicenseGroupUpdateHandler(entity));
    }

    @Override
    protected void deleteBefore(LicenseGroup entity, Boolean force) {
        deleteByIdBefore(entity.getLicenseGroupId(), force);
    }

    @Override
    protected void deleteByIdBefore(Long id, Boolean force) {
        LicenseSearchParams searchParams = new LicenseSearchParams();
        searchParams.setLicenseGroupId(id);

        if (licenseService.selectCount(searchParams) > 0) {
            throw new LicenseGroupCannotDeleteException();
        }
    }

    @Autowired
    public void setLicenseService(LicenseService licenseService) {
        this.licenseService = licenseService;
    }
}
