/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.web.controller.user;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.sns.GetUserInfoByCodeRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByUnionIdRequest;
import net.guerlab.sdk.dingtalk.response.sns.GetUserInfoByCodeResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByUnionIdResponse;
import net.guerlab.smart.dingtalk.web.controller.AbstractDingTalkControlPanelController;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.user.core.domain.UserDTO;
import net.guerlab.smart.user.core.entity.OauthLoginResponse;
import net.guerlab.smart.user.core.exception.NotOauthGroupAllowUserException;
import net.guerlab.smart.user.core.exception.UnsupportedLoginTypeException;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;

/**
 * 钉钉Oauth-控制面板
 *
 * @author guer
 */
@Tag(name = "钉钉Oauth-控制面板")
@RestController("/user/dingTalkOauth/controlPanel")
@RequestMapping("/user/dingTalkOauth/controlPanel")
public class DingTalkOauthControlPanelController extends AbstractDingTalkControlPanelController {

    @IgnoreLogin
    @Operation(summary = "通过qrCode登录")
    @GetMapping("/{appKey}/loginByQrCode")
    public OauthLoginResponse loginByQrCode(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "code", required = true) @RequestParam String code, HttpServletRequest request) {
        DingTalkClient client = dingTalkClientManagerService.getClient(appKey);
        if (client == null) {
            throw new UnsupportedLoginTypeException();
        }
        GetUserInfoByCodeResponse.UserInfo userInfo = getDingTalkUserInfo(client, code);
        String unionId = userInfo.getUnionId();
        if (unionId == null) {
            throw new NotOauthGroupAllowUserException();
        }
        String userId = getDingTalkUserId(client, unionId);
        String type = getOauthType(appKey);
        UserDTO user = findUser(client, type, userId);
        return getLoginSucceedDTO(user, userId, request, type);
    }

    private GetUserInfoByCodeResponse.UserInfo getDingTalkUserInfo(DingTalkClient client, String code) {
        GetUserInfoByCodeRequest request = new GetUserInfoByCodeRequest();
        request.setTmpAuthCode(code);

        GetUserInfoByCodeResponse response = client.execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        return response.getUserInfo();
    }

    private String getDingTalkUserId(DingTalkClient client, String unionId) {
        GetUserIdByUnionIdRequest request = new GetUserIdByUnionIdRequest();
        request.setUnionId(unionId);

        GetUserIdByUnionIdResponse response = client.execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        return response.getUserId();
    }

    @Override
    protected String getOauthTypePrefix() {
        return "DING_TALK_OAUTH";
    }
}
