
/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.web.controller.user;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.user.UserGetUserInfoRequest;
import net.guerlab.sdk.dingtalk.response.user.UserGetUserInfoResponse;
import net.guerlab.smart.dingtalk.web.controller.AbstractDingTalkControlPanelController;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.user.core.domain.UserDTO;
import net.guerlab.smart.user.core.entity.OauthLoginResponse;
import net.guerlab.smart.user.core.exception.NotOauthGroupAllowUserException;
import net.guerlab.smart.user.core.exception.UnsupportedLoginTypeException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;

/**
 * 钉钉-控制面板
 *
 * @author guer
 */
@Tag(name = "钉钉-控制面板")
@RestController("/user/dingTalk/controlPanel")
@RequestMapping("/user/dingTalk/controlPanel")
public class DingTalkControlPanelController extends AbstractDingTalkControlPanelController {

    @IgnoreLogin
    @Operation(summary = "通过code登录")
    @GetMapping("/{appKey}/loginByCode")
    @Transactional(rollbackFor = Exception.class)
    public OauthLoginResponse loginByCode(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "code", required = true) @RequestParam String code, HttpServletRequest request) {
        DingTalkClient client = dingTalkClientManagerService.getClient(appKey);
        if (client == null) {
            throw new UnsupportedLoginTypeException();
        }

        UserGetUserInfoResponse response = getDingTalkUserInfo(client, code);
        String userId = response.getUserId();
        if (StringUtils.isBlank(userId)) {
            throw new NotOauthGroupAllowUserException();
        }

        String type = getOauthType(appKey);
        UserDTO user = findUser(client, type, userId);
        return getLoginSucceedDTO(user, userId, request, type);
    }

    private UserGetUserInfoResponse getDingTalkUserInfo(DingTalkClient client, String code) {
        UserGetUserInfoRequest request = new UserGetUserInfoRequest();
        request.setCode(code);

        UserGetUserInfoResponse response = client.execute(request);
        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }
        return response;
    }

    @Override
    protected String getOauthTypePrefix() {
        return "DING_TALK";
    }
}
