/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.web.controller.user;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.dingtalk.core.domain.DingTalkAppDTO;
import net.guerlab.smart.dingtalk.core.enums.DingTalkAppType;
import net.guerlab.smart.dingtalk.core.exception.DingTalkAppInvalidException;
import net.guerlab.smart.dingtalk.core.searchparams.DingTalkAppSearchParams;
import net.guerlab.smart.dingtalk.service.entity.DingTalkApp;
import net.guerlab.smart.dingtalk.service.service.DingTalkAppService;
import net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.server.controller.BaseController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 钉钉应用
 *
 * @author guer
 */
@Tag(name = "钉钉应用")
@RestController("/user/dingTalkApp")
@RequestMapping("/user/dingTalkApp")
public class DingTalkAppController extends BaseController<DingTalkAppDTO, DingTalkApp, DingTalkAppService, DingTalkAppSearchParams, String> {

    private DingTalkClientManagerService managerService;

    private OperationLogApi operationLogApi;

    @Override
    protected ApplicationException nullPointException() {
        return new DingTalkAppInvalidException();
    }

    @Override
    public void copyProperties(DingTalkAppDTO dto, DingTalkApp entity, String id) {
        DingTalkAppType appType = entity.getAppType();
        super.copyProperties(dto, entity, id);
        if (id == null) {
            return;
        }
        entity.setAppKey(id);
        entity.setAppType(appType);
    }

    @Operation(description = "移除Token缓存", security = @SecurityRequirement(name = Constants.TOKEN))
    @PostMapping("/{appKey}/removeTokenCache")
    public void removeTokenCache(@Parameter(description = "appKey", required = true) @PathVariable String appKey) {
        DingTalkApp app = findOne0(appKey);

        managerService.getClient(appKey).getDingTalkConfigStorage().expireAccessToken();

        operationLogApi.add("移除Token缓存", UserContextHandler.getUserId(), app);
    }

    @Override
    public void afterSave(DingTalkApp entity, DingTalkAppDTO dto) {
        operationLogApi.add("添加钉钉应用", UserContextHandler.getUserId(), entity);
    }

    @Override
    public void afterUpdate(DingTalkApp entity, DingTalkAppDTO dto) {
        operationLogApi.add("编辑钉钉应用", UserContextHandler.getUserId(), entity);
    }

    @Override
    public void afterDelete(DingTalkApp entity) {
        operationLogApi.add("删除钉钉应用", UserContextHandler.getUserId(), entity.getAppKey());
    }

    @Autowired
    public void setManagerService(DingTalkClientManagerService managerService) {
        this.managerService = managerService;
    }

    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }

}
