/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.web.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.user.GetUserRequest;
import net.guerlab.sdk.dingtalk.response.user.GetUserResponse;
import net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService;
import net.guerlab.smart.dingtalk.web.domain.BindRequest;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.exception.PasswordErrorException;
import net.guerlab.smart.platform.commons.exception.ThirdPartyIdInvalidException;
import net.guerlab.smart.platform.commons.ip.IpUtils;
import net.guerlab.smart.user.api.OauthApi;
import net.guerlab.smart.user.api.UserApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import net.guerlab.smart.user.core.domain.OauthDTO;
import net.guerlab.smart.user.core.domain.UserDTO;
import net.guerlab.smart.user.core.entity.OauthLoginResponse;
import net.guerlab.smart.user.core.entity.UserInternalLoginRequest;
import net.guerlab.smart.user.core.exception.UnsupportedLoginTypeException;
import net.guerlab.smart.user.core.exception.UserHasBoundException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;

import javax.servlet.http.HttpServletRequest;
import java.util.Optional;

/**
 * 抽象钉钉控制面板
 *
 * @author guer
 */
public abstract class AbstractDingTalkControlPanelController {

    protected DingTalkClientManagerService dingTalkClientManagerService;

    protected OauthApi oauthApi;

    protected UserApi userApi;

    protected UserDTO findUser(DingTalkClient client, String type, String thirdPartyId) {
        Optional<OauthDTO> oauthOptional = oauthApi.findOneOptional(type, thirdPartyId);
        if (oauthOptional.isEmpty()) {
            return null;
        }

        OauthDTO oauth = oauthOptional.get();
        Long userId = oauth.getUserId();

        if (NumberHelper.greaterZero(userId)) {
            return userApi.findOne(oauth.getUserId());
        }

        GetUserResponse dingTalkUser = getDingTalkUser(client, thirdPartyId);
        String phone = StringUtils.trimToNull(dingTalkUser.getMobile());
        if (phone == null) {
            return null;
        }

        return userApi.findOneByPhoneOptional(phone).orElse(null);
    }

    protected UserDTO getUser(String type, String thirdPartyId) {
        Optional<OauthDTO> oauthOptional = oauthApi.findOneOptional(type, thirdPartyId);
        if (oauthOptional.isEmpty() || !NumberHelper.greaterZero(oauthOptional.get().getUserId())) {
            return null;
        }

        return userApi.findOne(oauthOptional.get().getUserId());
    }

    protected GetUserResponse getDingTalkUser(DingTalkClient client, String userId) {
        GetUserRequest request = new GetUserRequest();
        request.setUserId(userId);

        GetUserResponse response = client.execute(request);
        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }
        return response;
    }

    @IgnoreLogin
    @Operation(summary = "绑定")
    @PostMapping("/{appKey}/bind")
    public OauthLoginResponse bind(@Parameter(description = "type", required = true) @PathVariable String appKey,
            @RequestBody BindRequest bindRequest, HttpServletRequest request) {
        String type = getOauthType(appKey);
        String thirdPartyId = StringUtils.trimToNull(bindRequest.getThirdPartyId());
        if (thirdPartyId == null) {
            throw new ThirdPartyIdInvalidException();
        } else if (getUser(type, thirdPartyId) != null) {
            throw new UserHasBoundException();
        }

        UserDTO user = userApi.findOneByUsername(bindRequest.getUsername());
        if (oauthApi.findOneOptional(type, user.getUserId()).isPresent()) {
            throw new UserHasBoundException();
        }
        if (!userApi.checkPassword(user.getUserId(), bindRequest.getPassword())) {
            throw new PasswordErrorException();
        }

        oauthApi.bind(user.getUserId(), type, thirdPartyId);

        return getLoginSucceedDTO(user, thirdPartyId, request, type);
    }

    @Operation(summary = "解绑", security = @SecurityRequirement(name = Constants.TOKEN))
    @PostMapping("/{appKey}/unbind")
    public void bind(@Parameter(description = "type", required = true) @PathVariable String appKey) {
        oauthApi.unbind(UserContextHandler.getUserId(), getOauthType(appKey));
    }

    @Operation(summary = "获取绑定状态", security = @SecurityRequirement(name = Constants.TOKEN))
    @GetMapping("/{appKey}/bindStatus")
    public boolean bindStatus(@Parameter(description = "type", required = true) @PathVariable String appKey) {
        return oauthApi.findOneOptional(getOauthType(appKey), UserContextHandler.getUserId()).isPresent();
    }

    protected String getOauthType(String appKey) {
        appKey = StringUtils.trimToNull(appKey);
        if (appKey == null) {
            throw new UnsupportedLoginTypeException();
        }
        return getOauthTypePrefix() + "_" + appKey;
    }

    /**
     * 获取Oauth类型前缀
     *
     * @return Oauth类型前缀
     */
    protected abstract String getOauthTypePrefix();

    protected OauthLoginResponse getLoginSucceedDTO(UserDTO user, String openId, HttpServletRequest request,
            String loginType) {
        String ip = IpUtils.getIp(request);
        OauthLoginResponse oauthLoginResponse = new OauthLoginResponse();
        if (user != null) {
            UserInternalLoginRequest loginRequest = new UserInternalLoginRequest();
            loginRequest.setIp(ip);
            loginRequest.setUserId(user.getUserId());
            loginRequest.setLoginType(loginType);

            BeanUtils.copyProperties(userApi.internalLogin(loginRequest), oauthLoginResponse);
        }

        OauthDTO oauth = new OauthDTO();
        oauth.setUserId(user == null ? null : user.getUserId());
        oauth.setType(loginType);
        oauth.setThirdPartyId(openId);

        oauthLoginResponse.setThirdParty(oauth);

        return oauthLoginResponse;
    }

    @Autowired
    public void setDingTalkClientManagerService(DingTalkClientManagerService dingTalkClientManagerService) {
        this.dingTalkClientManagerService = dingTalkClientManagerService;
    }

    @Autowired
    public void setOauthApi(OauthApi oauthApi) {
        this.oauthApi = oauthApi;
    }

    @Autowired
    public void setUserApi(UserApi userApi) {
        this.userApi = userApi;
    }
}
