/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.internal.inside;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.sns.GetUserInfoByCodeRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByMobileRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByUnionIdRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserRequest;
import net.guerlab.sdk.dingtalk.request.user.UserGetUserInfoRequest;
import net.guerlab.sdk.dingtalk.response.sns.GetUserInfoByCodeResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByMobileResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByUnionIdResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserResponse;
import net.guerlab.sdk.dingtalk.response.user.UserGetUserInfoResponse;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserIdInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserOauthInfoDTO;
import net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService;
import net.guerlab.web.result.Result;
import net.guerlab.web.result.Succeed;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 钉钉客户端
 *
 * @author guer
 */
@Tag(name = "钉钉客户端")
@RestController("/inside/dingTalkClient")
@RequestMapping("/inside/dingTalkClient/{appKey}")
public class DingTalkClientController {

    private DingTalkClientManagerService managerService;

    @Operation(description = "获取accessToken")
    @GetMapping("/accessToken")
    public Result<String> getAccessToken(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey) {
        return new Succeed<>(Succeed.MSG, getClient(appKey).getAccessToken());
    }

    @Operation(description = "通过code获取用户信息")
    @GetMapping("/user/getUserInfo/{code}")
    public DingTalkUserInfoDTO getDingTalkUserInfo(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "code", required = true) @PathVariable String code) {
        UserGetUserInfoRequest request = new UserGetUserInfoRequest();
        request.setCode(code);

        UserGetUserInfoResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        DingTalkUserInfoDTO result = new DingTalkUserInfoDTO();
        result.setIsSys(response.isSys());
        result.setSysLevel(response.getSysLevel());
        result.setUserId(response.getUserId());

        return result;
    }

    @Operation(description = "通过code获取用户Oauth信息")
    @GetMapping("/user/getUserInfoByCode/{code}")
    public DingTalkUserOauthInfoDTO getDingTalkUserInfoByCode(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "code", required = true) @PathVariable String code) {
        GetUserInfoByCodeRequest request = new GetUserInfoByCodeRequest();
        request.setTmpAuthCode(code);

        GetUserInfoByCodeResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        GetUserInfoByCodeResponse.UserInfo userInfo = response.getUserInfo();

        DingTalkUserOauthInfoDTO result = new DingTalkUserOauthInfoDTO();
        result.setNick(userInfo.getNick());
        result.setOpenId(userInfo.getOpenId());
        result.setUnionId(userInfo.getUnionId());

        return result;
    }

    @Operation(description = "通过unionId获取用户id")
    @GetMapping("/user/getUserIdByUnionId/{unionId}")
    public DingTalkUserIdInfoDTO getUserIdByUnionId(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "unionId", required = true) @PathVariable String unionId) {
        GetUserIdByUnionIdRequest request = new GetUserIdByUnionIdRequest();
        request.setUnionId(unionId);

        GetUserIdByUnionIdResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        DingTalkUserIdInfoDTO result = new DingTalkUserIdInfoDTO();
        result.setUserId(response.getUserId());
        result.setContactType(response.getContactType());

        return result;
    }

    @Operation(description = "通过电话号码获取用户id")
    @GetMapping("/user/getUserIdByMobile/{mobile}")
    public Result<String> getUserIdByMobile(
            @Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "mobile", required = true) @PathVariable String mobile) {
        GetUserIdByMobileRequest request = new GetUserIdByMobileRequest();
        request.setMobile(mobile);

        GetUserIdByMobileResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        return new Succeed<>(Succeed.MSG, response.getUserId());
    }

    @Operation(description = "获取用户")
    @GetMapping("/user/getUser/{userId}")
    public DingTalkUserDTO getUser(@Parameter(description = "appKey", required = true) @PathVariable String appKey,
            @Parameter(description = "userId", required = true) @PathVariable String userId) {
        GetUserRequest request = new GetUserRequest();
        request.setUserId(userId);

        GetUserResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        DingTalkUserDTO result = new DingTalkUserDTO();
        BeanUtils.copyProperties(response, result);

        return result;
    }

    private DingTalkClient getClient(String appKey) {
        return managerService.getClient(appKey);
    }

    @Autowired
    public void setManagerService(DingTalkClientManagerService managerService) {
        this.managerService = managerService;
    }
}
