package net.guerlab.smart.dingtalk.internal.inside;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.sns.GetUserInfoByCodeRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByMobileRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByUnionIdRequest;
import net.guerlab.sdk.dingtalk.request.user.UserGetUserInfoRequest;
import net.guerlab.sdk.dingtalk.response.sns.GetUserInfoByCodeResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByMobileResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByUnionIdResponse;
import net.guerlab.sdk.dingtalk.response.user.UserGetUserInfoResponse;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserIdInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserOauthInfoDTO;
import net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService;
import net.guerlab.web.result.Result;
import net.guerlab.web.result.Succeed;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 钉钉客户端
 *
 * @author guer
 */
@RestController("/inside/dingTalkClient")
@RequestMapping("/inside/dingTalkClient/{appKey}")
public class DingTalkClientController {

    private DingTalkClientManagerService managerService;

    @ApiOperation("获取accessToken")
    @GetMapping("/accessToken")
    public Result<String> getAccessToken(@ApiParam(value = "appKey", required = true) @PathVariable String appKey) {
        return new Succeed<>(Succeed.MSG, getClient(appKey).getAccessToken());
    }

    @ApiOperation("通过code获取用户信息")
    @GetMapping("/user/getuserinfo/{code}")
    public DingTalkUserInfoDTO getDingTalkUserInfo(@ApiParam(value = "appKey", required = true) @PathVariable String appKey,
            @ApiParam(value = "code", required = true) @PathVariable String code) {
        UserGetUserInfoRequest request = new UserGetUserInfoRequest();
        request.setCode(code);

        UserGetUserInfoResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        DingTalkUserInfoDTO result = new DingTalkUserInfoDTO();
        result.setIsSys(response.isSys());
        result.setSysLevel(response.getSysLevel());
        result.setUserId(response.getUserId());

        return result;
    }

    @ApiOperation("通过code获取用户Oauth信息")
    @GetMapping("/user/getuserinfo_bycode/{code}")
    public DingTalkUserOauthInfoDTO getDingTalkUserInfoByCode(@ApiParam(value = "appKey", required = true) @PathVariable String appKey,
            @ApiParam(value = "code", required = true) @PathVariable String code) {
        GetUserInfoByCodeRequest request = new GetUserInfoByCodeRequest();
        request.setTmpAuthCode(code);

        GetUserInfoByCodeResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        GetUserInfoByCodeResponse.UserInfo userInfo = response.getUserInfo();

        DingTalkUserOauthInfoDTO result = new DingTalkUserOauthInfoDTO();
        result.setNick(userInfo.getNick());
        result.setOpenId(userInfo.getOpenId());
        result.setUnionId(userInfo.getUnionId());

        return result;
    }

    @ApiOperation("通过unionId获取用户id")
    @GetMapping("/user/getUserIdByUnionId/{unionId}")
    public DingTalkUserIdInfoDTO getUserIdByUnionId(
            @ApiParam(value = "appKey", required = true) @PathVariable String appKey,
            @ApiParam(value = "unionId", required = true) @PathVariable String unionId) {
        GetUserIdByUnionIdRequest request = new GetUserIdByUnionIdRequest();
        request.setUnionId(unionId);

        GetUserIdByUnionIdResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        DingTalkUserIdInfoDTO result = new DingTalkUserIdInfoDTO();
        result.setUserId(response.getUserId());
        result.setContactType(response.getContactType());

        return result;
    }

    @ApiOperation("通过电话号码获取用户id")
    @GetMapping("/user/getUserIdByMobile/{mobile}")
    public Result<String> getUserIdByMobile(@ApiParam(value = "appKey", required = true) @PathVariable String appKey,
            @ApiParam(value = "mobile", required = true) @PathVariable String mobile) {
        GetUserIdByMobileRequest request = new GetUserIdByMobileRequest();
        request.setMobile(mobile);

        GetUserIdByMobileResponse response = getClient(appKey).execute(request);

        if (response.getErrcode() > 0) {
            throw new ApplicationException(response.getErrmsg(), response.getErrcode());
        }

        return new Succeed<>(Succeed.MSG, response.getUserId());
    }

    private DingTalkClient getClient(String appKey) {
        return managerService.getClient(appKey);
    }

    @Autowired
    public void setManagerService(DingTalkClientManagerService managerService) {
        this.managerService = managerService;
    }
}
