/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.sdk.dingtalk.client.DingTalkClient;
import net.guerlab.sdk.dingtalk.request.sns.GetUserInfoByCodeRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByMobileRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserIdByUnionIdRequest;
import net.guerlab.sdk.dingtalk.request.user.GetUserRequest;
import net.guerlab.sdk.dingtalk.request.user.UserGetUserInfoRequest;
import net.guerlab.sdk.dingtalk.response.sns.GetUserInfoByCodeResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByMobileResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserIdByUnionIdResponse;
import net.guerlab.sdk.dingtalk.response.user.GetUserResponse;
import net.guerlab.sdk.dingtalk.response.user.UserGetUserInfoResponse;
import net.guerlab.smart.dingtalk.api.DingTalkClientApi;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserIdInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserOauthInfoDTO;
import net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.lang.NonNull;

/**
 * 钉钉请求客户端管理本地化调用自动配置
 *
 * @author guer
 */
@Configuration
@Conditional(DingTalkClientApiLocalServiceAutoConfigure.WrapperCondition.class)
public class DingTalkClientApiLocalServiceAutoConfigure {

    @Bean
    @ConditionalOnBean(DingTalkClientManagerService.class)
    public DingTalkClientApi dingTalkClientApiLocalServiceWrapper(DingTalkClientManagerService managerService) {
        return new DingTalkClientApiLocalServiceWrapper(managerService);
    }

    @SuppressWarnings("WeakerAccess")
    static class WrapperCondition implements Condition {

        @Override
        public boolean matches(@NonNull ConditionContext context, @NonNull AnnotatedTypeMetadata metadata) {
            try {
                return WrapperCondition.class.getClassLoader().loadClass("net.guerlab.smart.dingtalk.service.service.DingTalkClientManagerService") != null;
            } catch (Exception e) {
                return false;
            }
        }
    }

    @AllArgsConstructor
    private static class DingTalkClientApiLocalServiceWrapper implements DingTalkClientApi {

        private final DingTalkClientManagerService managerService;

        @Override
        public String getAccessToken(String appKey) {
            return getClient(appKey).getAccessToken();
        }

        @Override
        public DingTalkUserInfoDTO getDingTalkUserInfo(String appKey, String code) {
            UserGetUserInfoRequest request = new UserGetUserInfoRequest();
            request.setCode(code);

            UserGetUserInfoResponse response = getClient(appKey).execute(request);

            if (response.getErrcode() > 0) {
                throw new ApplicationException(response.getErrmsg(), response.getErrcode());
            }

            DingTalkUserInfoDTO result = new DingTalkUserInfoDTO();
            result.setIsSys(response.isSys());
            result.setSysLevel(response.getSysLevel());
            result.setUserId(response.getUserId());

            return result;
        }

        @Override
        public DingTalkUserOauthInfoDTO getDingTalkUserInfoByCode(String appKey, String code) {
            GetUserInfoByCodeRequest request = new GetUserInfoByCodeRequest();
            request.setTmpAuthCode(code);

            GetUserInfoByCodeResponse response = getClient(appKey).execute(request);

            if (response.getErrcode() > 0) {
                throw new ApplicationException(response.getErrmsg(), response.getErrcode());
            }

            GetUserInfoByCodeResponse.UserInfo userInfo = response.getUserInfo();

            DingTalkUserOauthInfoDTO result = new DingTalkUserOauthInfoDTO();
            result.setNick(userInfo.getNick());
            result.setOpenId(userInfo.getOpenId());
            result.setUnionId(userInfo.getUnionId());

            return result;
        }

        @Override
        public DingTalkUserIdInfoDTO getUserIdByUnionId(String appKey, String unionId) {
            GetUserIdByUnionIdRequest request = new GetUserIdByUnionIdRequest();
            request.setUnionId(unionId);

            GetUserIdByUnionIdResponse response = getClient(appKey).execute(request);

            if (response.getErrcode() > 0) {
                throw new ApplicationException(response.getErrmsg(), response.getErrcode());
            }

            DingTalkUserIdInfoDTO result = new DingTalkUserIdInfoDTO();
            result.setUserId(response.getUserId());
            result.setContactType(response.getContactType());

            return result;
        }

        @Override
        public String getUserIdByMobile(String appKey, String mobile) {
            GetUserIdByMobileRequest request = new GetUserIdByMobileRequest();
            request.setMobile(mobile);

            GetUserIdByMobileResponse response = getClient(appKey).execute(request);

            if (response.getErrcode() > 0) {
                throw new ApplicationException(response.getErrmsg(), response.getErrcode());
            }

            return response.getUserId();
        }

        @Override
        public DingTalkUserDTO getUser(String appKey, String userId) {
            GetUserRequest request = new GetUserRequest();
            request.setUserId(userId);

            GetUserResponse response = getClient(appKey).execute(request);

            if (response.getErrcode() > 0) {
                throw new ApplicationException(response.getErrmsg(), response.getErrcode());
            }

            DingTalkUserDTO result = new DingTalkUserDTO();
            BeanUtils.copyProperties(response, result);

            return result;
        }

        private DingTalkClient getClient(String appKey) {
            return managerService.getClient(appKey);
        }
    }

}
