/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.commons.exception.ApplicationException;
import net.guerlab.smart.dingtalk.api.DingTalkClientApi;
import net.guerlab.smart.dingtalk.api.feign.FeignDingTalkClientApi;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserIdInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserInfoDTO;
import net.guerlab.smart.dingtalk.core.domain.DingTalkUserOauthInfoDTO;
import net.guerlab.web.result.Result;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * 钉钉请求客户端管理远程调用自动配置
 *
 * @author guer
 */
@Configuration
@AutoConfigureAfter(DingTalkClientApiLocalServiceAutoConfigure.class)
public class DingTalkClientApiFeignAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnMissingBean(DingTalkClientApi.class)
    public DingTalkClientApi dingTalkClientApiFeignWrapper(FeignDingTalkClientApi api) {
        return new DingTalkClientApiFeignWrapper(api);
    }

    @AllArgsConstructor
    private static class DingTalkClientApiFeignWrapper implements DingTalkClientApi {

        private final FeignDingTalkClientApi api;

        @Override
        public String getAccessToken(String appKey) {
            Result<String> result = api.getAccessToken(appKey);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public DingTalkUserInfoDTO getDingTalkUserInfo(String appKey, String code) {
            Result<DingTalkUserInfoDTO> result = api.getDingTalkUserInfo(appKey, code);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public DingTalkUserOauthInfoDTO getDingTalkUserInfoByCode(String appKey, String code) {
            Result<DingTalkUserOauthInfoDTO> result = api.getDingTalkUserInfoByCode(appKey, code);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public DingTalkUserIdInfoDTO getUserIdByUnionId(String appKey, String unionId) {
            Result<DingTalkUserIdInfoDTO> result = api.getUserIdByUnionId(appKey, unionId);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public String getUserIdByMobile(String appKey, String mobile) {
            Result<String> result = api.getUserIdByMobile(appKey, mobile);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }

        @Override
        public DingTalkUserDTO getUser(String appKey, String userId) {
            Result<DingTalkUserDTO> result = api.getUser(appKey, userId);

            if (result.isStatus()) {
                return result.getData();
            }

            throw new ApplicationException(result.getMessage(), result.getErrorCode());
        }
    }

}
