/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.dingtalk.api.autoconfig;

import lombok.AllArgsConstructor;
import net.guerlab.smart.dingtalk.api.DingTalkAppApi;
import net.guerlab.smart.dingtalk.api.feign.FeignDingTalkAppApi;
import net.guerlab.smart.dingtalk.core.domain.DingTalkAppDTO;
import net.guerlab.smart.dingtalk.core.exception.DingTalkAppInvalidException;
import net.guerlab.smart.dingtalk.core.searchparams.DingTalkAppSearchParams;
import net.guerlab.spring.searchparams.SearchParamsUtils;
import net.guerlab.web.result.ListObject;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.*;

/**
 * 钉钉应用服务远程调用自动配置
 *
 * @author guer
 */
@Configuration
@AutoConfigureAfter(DingTalkAppApiLocalServiceAutoConfigure.class)
public class DingTalkAppApiFeignAutoConfigure {

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Bean
    @ConditionalOnMissingBean(DingTalkAppApi.class)
    public DingTalkAppApi dingTalkAppApiFeignWrapper(FeignDingTalkAppApi api) {
        return new DingTalkAppApiFeignWrapper(api);
    }

    @AllArgsConstructor
    private static class DingTalkAppApiFeignWrapper implements DingTalkAppApi {

        private final FeignDingTalkAppApi api;

        @Override
        public DingTalkAppDTO findOne(String appId) {
            return Optional.ofNullable(api.findOne(appId).getData()).orElseThrow(DingTalkAppInvalidException::new);
        }

        @Override
        public ListObject<DingTalkAppDTO> findList(DingTalkAppSearchParams searchParams) {
            Map<String, Object> params = new HashMap<>(8);
            SearchParamsUtils.handler(searchParams, params);
            return Optional.ofNullable(api.findList(params).getData()).orElse(ListObject.empty());
        }

        @Override
        public List<DingTalkAppDTO> findAll(DingTalkAppSearchParams searchParams) {
            Map<String, Object> params = new HashMap<>(8);
            SearchParamsUtils.handler(searchParams, params);
            return Optional.ofNullable(api.findAll(params).getData()).orElse(Collections.emptyList());
        }
    }

}
