package net.guerlab.smart.article.web.controller.user;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.smart.article.core.ArticleConstants;
import net.guerlab.smart.article.core.domain.ArticleCategoryDTO;
import net.guerlab.smart.article.core.domain.ArticleDTO;
import net.guerlab.smart.article.core.enums.AuditStatus;
import net.guerlab.smart.article.core.enums.PublishType;
import net.guerlab.smart.article.core.exception.ArticleAuditStatusErrorException;
import net.guerlab.smart.article.core.exception.ArticleInvalidException;
import net.guerlab.smart.article.core.searchparams.ArticleCategorySearchParams;
import net.guerlab.smart.article.service.entity.Article;
import net.guerlab.smart.article.service.entity.ArticleCategory;
import net.guerlab.smart.article.service.service.ArticleConfigService;
import net.guerlab.smart.article.service.service.ArticleService;
import net.guerlab.smart.article.web.controller.AbstractArticleController;
import net.guerlab.smart.user.api.OperationLogApi;
import net.guerlab.smart.user.api.UserApi;
import net.guerlab.smart.user.auth.UserContextHandler;
import net.guerlab.smart.user.auth.annotation.HasPermission;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 文章
 *
 * @author guer
 */
@Api(tags = "文章")
@RestController("/user/article")
@RequestMapping("/user/article")
public class ArticleController extends AbstractArticleController<Article, ArticleService> {

    private ArticleConfigService configService;

    private OperationLogApi operationLogApi;

    private UserApi userApi;

    @ApiOperation("添加")
    @PostMapping
    public ArticleDTO save(@ApiParam(value = "对象数据", required = true) @RequestBody ArticleDTO dto) {
        Article entity = new Article();
        BeanUtils.copyProperties(dto, entity);

        if (needAudit()) {
            entity.setAuditStatus(AuditStatus.WAIT);
        } else {
            entity.setAuditStatus(AuditStatus.PASS);
        }

        getService().insert(entity);
        operationLogApi.add("添加文章", UserContextHandler.getUserId(), entity);
        return entity.toDTO();
    }

    @ApiOperation("编辑")
    @PutMapping("/{id}")
    public ArticleDTO update(@ApiParam(value = "id", required = true) @PathVariable Long id,
            @ApiParam(value = "对象数据", required = true) @RequestBody ArticleDTO dto) {
        Article entity = getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);

        entity.setArticleCategoryId(dto.getArticleCategoryId());
        entity.setSecondaryArticleCategoryIds(dto.getSecondaryArticleCategoryIds());
        entity.setTitle(dto.getTitle());
        entity.setCoverUrl(dto.getCoverUrl());
        entity.setAuthor(dto.getAuthor());
        entity.setReleaseTime(dto.getReleaseTime());
        entity.setSynopsis(dto.getSynopsis());
        entity.setContent(dto.getContent());
        entity.setOriginalLink(dto.getOriginalLink());
        entity.setAlwaysRedirect(dto.getAlwaysRedirect());
        entity.setPublishType(dto.getPublishType());
        entity.setPlanPublishTime(dto.getPlanPublishTime());
        entity.setAttachments(dto.getAttachments());
        entity.setOrderNum(dto.getOrderNum());

        if (needAudit()) {
            entity.setAuditStatus(AuditStatus.WAIT);
        } else {
            entity.setAuditStatus(AuditStatus.PASS);
        }

        getService().updateSelectiveById(entity);
        operationLogApi.add("编辑文章", UserContextHandler.getUserId(), entity);
        return getService().selectById(id).toDTO();
    }

    @ApiOperation("显示")
    @PutMapping("/{id}/show")
    public ArticleDTO show(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        Article entity = getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);
        if (entity.getAuditStatus() != AuditStatus.PASS) {
            throw new ArticleAuditStatusErrorException();
        }
        entity.setPublished(true);
        entity.setPublishType(PublishType.MANUAL);
        entity.setPublishTime(LocalDateTime.now());
        getService().updateSelectiveById(entity);
        operationLogApi.add("显示文章", UserContextHandler.getUserId(), id);
        return getService().selectById(id).toDTO();
    }

    @ApiOperation("不显示")
    @PutMapping("/{id}/unShow")
    public ArticleDTO unShow(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        Article entity = getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);
        entity.setPublished(false);
        entity.setPublishType(PublishType.MANUAL);
        getService().updateSelectiveById(entity);
        operationLogApi.add("隐藏文章", UserContextHandler.getUserId(), id);
        return getService().selectById(id).toDTO();
    }

    @ApiOperation("删除")
    @DeleteMapping("/{id}")
    public void delete(@ApiParam(value = "id", required = true) @PathVariable Long id,
            @ApiParam(value = "强制删除标志") @RequestParam(required = false) Boolean force) {
        getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);
        getService().deleteById(id, force);
        operationLogApi.add("删除文章", UserContextHandler.getUserId(), id);
    }

    @HasPermission(ArticleConstants.ARTICLE_AUDIT_PERMISSION)
    @ApiOperation("审核通过")
    @PutMapping("/{id}/audit/pass")
    public ArticleDTO auditPass(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        Article entity = getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);
        if (entity.getAuditStatus() != AuditStatus.WAIT) {
            throw new ArticleAuditStatusErrorException();
        }
        if (entity.getPublishType() == PublishType.AUTOMATIC) {
            entity.setPublished(true);
        } else if (entity.getPublishType() == PublishType.TIMING) {
            entity.setPublished(entity.getPlanPublishTime().isBefore(LocalDateTime.now()));
        }
        entity.setAuditStatus(AuditStatus.PASS);
        getService().updateSelectiveById(entity);
        operationLogApi.add("文章审核通过", UserContextHandler.getUserId(), id);
        return getService().selectById(id).toDTO();
    }

    @HasPermission(ArticleConstants.ARTICLE_AUDIT_PERMISSION)
    @ApiOperation("审核拒绝")
    @PutMapping("/{id}/audit/refuse")
    public ArticleDTO auditRefuse(@ApiParam(value = "id", required = true) @PathVariable Long id) {
        Article entity = getService().selectByIdOptional(id).orElseThrow(ArticleInvalidException::new);
        if (entity.getAuditStatus() != AuditStatus.WAIT) {
            throw new ArticleAuditStatusErrorException();
        }
        entity.setPublished(false);
        entity.setAuditStatus(AuditStatus.REFUSE);
        getService().updateSelectiveById(entity);
        operationLogApi.add("文章审核拒绝", UserContextHandler.getUserId(), id);
        return getService().selectById(id).toDTO();
    }

    @Override
    protected Article findOne0(String id) {
        Article article = super.findOne0(id);
        ArticleCategorySearchParams categorySearchParams = new ArticleCategorySearchParams();
        categorySearchParams.setArticleCategoryIds(article.getArticleCategoryIds());

        article.setCategories(getCategoryService().selectAll(categorySearchParams));

        return article;
    }

    @Override
    protected void fillCategories(Collection<ArticleDTO> list) {
        if (list == null || list.isEmpty()) {
            return;
        }

        Collection<Long> allCategoryIds = list.stream().map(ArticleDTO::getArticleCategoryIds)
                .flatMap(Collection::stream).collect(Collectors.toSet());

        if (allCategoryIds.isEmpty()) {
            return;
        }

        ArticleCategorySearchParams categorySearchParams = new ArticleCategorySearchParams();
        categorySearchParams.setArticleCategoryIds(allCategoryIds);

        Map<Long, ArticleCategoryDTO> categoryMap = CollectionUtil
                .toMap(getCategoryService().selectAll(categorySearchParams), ArticleCategory::getArticleCategoryId,
                        ArticleCategory::toDTO);

        list.forEach((article) -> article.setCategories(
                article.getArticleCategoryIds().stream().map(categoryMap::get).filter(Objects::nonNull)
                        .collect(Collectors.toCollection(LinkedList::new))));
    }

    private boolean needAudit() {
        return enableAudit() && notHasAuditPermission();
    }

    private boolean enableAudit() {
        return configService.match("audit", "true");
    }

    private boolean notHasAuditPermission() {
        return !userApi.hasPermission(UserContextHandler.getUserId(),
                Collections.singletonList(ArticleConstants.ARTICLE_AUDIT_PERMISSION));
    }

    @Autowired
    public void setConfigService(ArticleConfigService configService) {
        this.configService = configService;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setOperationLogApi(OperationLogApi operationLogApi) {
        this.operationLogApi = operationLogApi;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setUserApi(UserApi userApi) {
        this.userApi = userApi;
    }
}
