package net.guerlab.smart.article.web.controller;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.smart.article.core.domain.ArticleDTO;
import net.guerlab.smart.article.core.exception.ArticleInvalidException;
import net.guerlab.smart.article.core.searchparams.ArticleSearchParams;
import net.guerlab.smart.article.service.service.AbstractArticleService;
import net.guerlab.smart.article.service.service.ArticleCategoryService;
import net.guerlab.smart.platform.commons.util.BeanConvertUtils;
import net.guerlab.spring.commons.dto.ConvertDTO;
import net.guerlab.web.result.ListObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;

import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

/**
 * 抽象文章查询
 *
 * @author guer
 */
public abstract class AbstractArticleController<E extends ConvertDTO<ArticleDTO>, S extends AbstractArticleService<E>> {

    private S service;

    private ArticleCategoryService categoryService;

    @ApiOperation("查询详情")
    @GetMapping("/{id}")
    public ArticleDTO findOne(@ApiParam(value = "文章ID/唯一key", required = true) @PathVariable String id) {
        return findOne0(id).toDTO();
    }

    protected E findOne0(String id) {
        E article;
        if (Pattern.matches(AbstractArticleService.NUMBER_REG, id)) {
            article = getService().selectById(Long.parseLong(id));
        } else {
            article = getService().selectByUniqueKey(id);
        }

        if (article == null) {
            throw new ArticleInvalidException();
        }
        return article;
    }

    @ApiOperation("查询列表")
    @GetMapping
    public ListObject<ArticleDTO> findList(ArticleSearchParams searchParams) {
        beforeFindList(searchParams);
        ListObject<ArticleDTO> result = BeanConvertUtils.toListObject(getService().queryPage(searchParams));
        if (searchParams.getQueryCategory() != null && searchParams.getQueryCategory()) {
            fillCategories(result.getList());
        }
        return result;
    }

    @ApiOperation("查询全部")
    @GetMapping("/all")
    public List<ArticleDTO> findAll(ArticleSearchParams searchParams) {
        beforeFindList(searchParams);
        List<ArticleDTO> result = BeanConvertUtils.toList(getService().queryAll(searchParams));
        if (searchParams.getQueryCategory() != null && searchParams.getQueryCategory()) {
            fillCategories(result);
        }
        return result;
    }

    protected void beforeFindList(ArticleSearchParams searchParams) {

    }

    protected void fillCategories(Collection<ArticleDTO> list) {

    }

    public S getService() {
        return this.service;
    }

    public ArticleCategoryService getCategoryService() {
        return categoryService;
    }

    @Autowired
    public void setCategoryService(ArticleCategoryService categoryService) {
        this.categoryService = categoryService;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setService(S service) {
        this.service = service;
    }
}
