/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.article.service.service.impl;

import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.smart.article.core.searchparams.ArticleConfigSearchParams;
import net.guerlab.smart.article.service.entity.ArticleConfig;
import net.guerlab.smart.article.service.mapper.ArticleConfigMapper;
import net.guerlab.smart.article.service.service.ArticleConfigService;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 文章配置服务实现
 *
 * @author guer
 */
@Service
public class ArticleConfigServiceImpl extends BaseServiceImpl<ArticleConfig, String, ArticleConfigMapper, ArticleConfigSearchParams>
        implements ArticleConfigService {

    @Override
    public String findOne(String key) {
        key = StringUtils.trimToNull(key);

        if (key == null) {
            return null;
        }

        ArticleConfig config = getBaseMapper().selectById(key);
        return config == null ? null : config.getConfigValue();
    }

    @Override
    public Map<String, String> findMap(Collection<String> keys) {
        List<String> keyList = keysFilter(keys);

        if (keyList.isEmpty()) {
            return Collections.emptyMap();
        }

        ArticleConfigSearchParams searchParams = new ArticleConfigSearchParams();
        searchParams.setConfigKeys(keyList);

        return CollectionUtil.toMap(selectAll(searchParams), ArticleConfig::getConfigKey, ArticleConfig::getConfigValue);
    }

    @Override
    public Map<String, String> findAll() {
        return CollectionUtil.toMap(selectAll(), ArticleConfig::getConfigKey, ArticleConfig::getConfigValue);
    }

    @Override
    public void save(Map<String, String> configs) {
        if (configs == null || configs.isEmpty()) {
            return;
        }

        List<ArticleConfig> configList = configs.entrySet().stream().map(this::buildConfig).filter(Objects::nonNull).collect(Collectors.toList());

        if (configList.isEmpty()) {
            return;
        }

        getBaseMapper().replaceInsertList(configList);
    }

    @Override
    public void delete(Collection<String> keys) {
        List<String> keyList = keysFilter(keys);

        if (keyList.isEmpty()) {
            return;
        }

        ArticleConfigSearchParams searchParams = new ArticleConfigSearchParams();
        searchParams.setConfigKeys(keyList);

        getBaseMapper().delete(getQueryWrapper(searchParams));
    }

    private List<String> keysFilter(Collection<String> keys) {
        if (keys == null || keys.isEmpty()) {
            return Collections.emptyList();
        }

        return keys.stream().map(StringUtils::trimToNull).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private ArticleConfig buildConfig(Map.Entry<String, String> entry) {
        if (entry == null) {
            return null;
        }

        String key = StringUtils.trimToNull(entry.getKey());
        String value = StringUtils.trimToNull(entry.getValue());

        if (key == null || key.length() > KEY_MAX_LENGTH || value == null || value.length() > VALUE_MAX_LENGTH) {
            return null;
        }

        ArticleConfig config = new ArticleConfig();
        config.setConfigKey(key);
        config.setConfigValue(value);

        return config;
    }

    @Override
    public Class<ArticleConfig> getEntityClass() {
        return ArticleConfig.class;
    }
}
