/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.article.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.article.core.exception.*;
import net.guerlab.smart.article.core.searchparams.ArticleCategorySearchParams;
import net.guerlab.smart.article.service.entity.ArticleCategory;
import net.guerlab.smart.article.service.handlers.ArticleCategoryDeleteAfterHandler;
import net.guerlab.smart.article.service.handlers.ArticleCategoryUpdateAfterHandler;
import net.guerlab.smart.article.service.mapper.ArticleCategoryMapper;
import net.guerlab.smart.article.service.service.ArticleCategoryService;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.util.OrderEntityUtils;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.spring.commons.util.SpringApplicationContextUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Optional;

/**
 * 文章分类服务实现
 *
 * @author guer
 */
@Service
public class ArticleCategoryServiceImpl extends BaseServiceImpl<ArticleCategory, Long, ArticleCategoryMapper, ArticleCategorySearchParams>
        implements ArticleCategoryService {

    @Override
    protected void insertBefore(ArticleCategory entity) {
        String name = StringUtils.trimToNull(entity.getArticleCategoryName());
        String type = StringUtils.trimToNull(entity.getArticleCategoryType());
        Long parentId = entity.getParentId();

        String listPageType = StringUtils.trimToEmpty(entity.getListPageType());
        String detailPageType = StringUtils.trimToEmpty(entity.getDetailPageType());

        if (name == null) {
            throw new ArticleCategoryNameInvalidException();
        } else if (name.length() > CATEGORY_NAME_MAX_LENGTH) {
            throw new ArticleCategoryNameLengthErrorException();
        }

        if (NumberHelper.greaterZero(parentId)) {
            if (selectById(parentId) == null) {
                throw new ParentArticleCategoryInvalidException();
            }
        } else {
            entity.setParentId(Constants.DEFAULT_PARENT_ID);
        }

        if (listPageType.length() > PAGE_TYPE_MAX_LENGTH) {
            throw new ListPageTypeLengthErrorException();
        }
        if (detailPageType.length() > PAGE_TYPE_MAX_LENGTH) {
            throw new DetailPageTypeLengthErrorException();
        }

        entity.setArticleCategoryId(sequence.nextId());
        entity.setArticleCategoryName(name);
        entity.setArticleCategoryType(Optional.ofNullable(type).orElse(ArticleCategory.DEFAULT_ARTICLE_CATEGORY_TYPE));
        if (entity.getEnabled() == null) {
            entity.setEnabled(true);
        }
        if (entity.getDisplay() == null) {
            entity.setDisplay(true);
        }
        entity.setListPageType(listPageType);
        entity.setDetailPageType(detailPageType);

        OrderEntityUtils.propertiesCheck(entity);
    }

    @Override
    protected void updateBefore(ArticleCategory entity) {
        String name = StringUtils.trimToNull(entity.getArticleCategoryName());
        if (name != null && name.length() > CATEGORY_NAME_MAX_LENGTH) {
            throw new ArticleCategoryNameLengthErrorException();
        }
        entity.setArticleCategoryName(name);

        String listPageType = StringUtils.trimToNull(entity.getListPageType());
        if (listPageType != null && listPageType.length() > PAGE_TYPE_MAX_LENGTH) {
            throw new ListPageTypeLengthErrorException();
        }

        String detailPageType = StringUtils.trimToNull(entity.getDetailPageType());
        if (detailPageType != null && detailPageType.length() > PAGE_TYPE_MAX_LENGTH) {
            throw new DetailPageTypeLengthErrorException();
        }
    }

    @Override
    protected void updateAfter(ArticleCategory entity) {
        SpringApplicationContextUtil.getContext().getBeansOfType(ArticleCategoryUpdateAfterHandler.class).values().forEach(handler -> handler.articleCategoryUpdateAfterHandler(entity));
    }

    @Override
    protected void deleteAfter(ArticleCategorySearchParams searchParams, Boolean force) {
        deleteByIdAfter(searchParams.getArticleCategoryId(), force);
    }

    @Override
    protected void deleteByIdAfter(Long id, Boolean force) {
        SpringApplicationContextUtil.getContext().getBeansOfType(ArticleCategoryDeleteAfterHandler.class).values().forEach(handler -> handler.articleCategoryDeleteAfterHandler(id));
    }
}
