/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.article.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.article.core.searchparams.ArticleCategoryMappingSearchParams;
import net.guerlab.smart.article.service.entity.ArticleCategoryMapping;
import net.guerlab.smart.article.service.handlers.ArticleCategoryDeleteAfterHandler;
import net.guerlab.smart.article.service.mapper.ArticleCategoryMappingMapper;
import net.guerlab.smart.article.service.service.ArticleCategoryMappingService;
import net.guerlab.spring.searchparams.SearchParamsUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 文章分类映射服务实现
 *
 * @author guer
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class ArticleCategoryMappingServiceImpl implements ArticleCategoryMappingService, ArticleCategoryDeleteAfterHandler {

    private ArticleCategoryMappingMapper mapper;

    @Override
    public Collection<ArticleCategoryMapping> findList(ArticleCategoryMappingSearchParams searchParams) {
        return mapper.selectList(getQueryWrapper(searchParams));
    }

    @Override
    public void save(Collection<ArticleCategoryMapping> list) {
        if (list == null || list.isEmpty()) {
            return;
        }

        List<ArticleCategoryMapping> waitSaves = list.stream().filter(this::saveFilter).collect(Collectors.toList());

        if (waitSaves.isEmpty()) {
            return;
        }

        mapper.replaceInsertList(waitSaves);
    }

    private boolean saveFilter(ArticleCategoryMapping item) {
        return item != null && NumberHelper.allGreaterZero(item.getArticleId(), item.getArticleCategoryId());
    }

    @Override
    public void delete(ArticleCategoryMappingSearchParams searchParams) {
        mapper.delete(getQueryWrapper(searchParams));
    }

    @Override
    public void articleCategoryDeleteAfterHandler(Long articleCategoryId) {
        if (!NumberHelper.greaterZero(articleCategoryId)) {
            return;
        }

        ArticleCategoryMappingSearchParams searchParams = new ArticleCategoryMappingSearchParams();
        searchParams.setArticleCategoryId(articleCategoryId);

        delete(searchParams);
    }

    private QueryWrapper<ArticleCategoryMapping> getQueryWrapper(ArticleCategoryMappingSearchParams searchParams) {
        QueryWrapper<ArticleCategoryMapping> wrapper = new QueryWrapper<>();
        SearchParamsUtils.handler(searchParams, wrapper);
        return wrapper;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setMapper(ArticleCategoryMappingMapper mapper) {
        this.mapper = mapper;
    }
}
