/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.article.service.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.article.core.domain.ArticleDTO;
import net.guerlab.smart.article.core.searchparams.ArticleCategoryMappingSearchParams;
import net.guerlab.smart.article.core.searchparams.ArticleCategorySearchParams;
import net.guerlab.smart.article.core.searchparams.ArticleSearchParams;
import net.guerlab.smart.article.service.entity.ArticleCategory;
import net.guerlab.smart.article.service.entity.ArticleCategoryMapping;
import net.guerlab.smart.article.service.service.AbstractArticleService;
import net.guerlab.smart.article.service.service.ArticleCategoryMappingService;
import net.guerlab.smart.article.service.service.ArticleCategoryService;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.spring.commons.dto.Convert;
import net.guerlab.web.result.ListObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Collection;
import java.util.Collections;

/**
 * 抽象文章服务实现
 *
 * @param <E>
 *         文章类型
 * @author guer
 */
public abstract class AbstractArticleServiceImpl<E extends Convert<ArticleDTO>, M extends BaseMapper<E>> extends BaseServiceImpl<E, Long, M, ArticleSearchParams> implements AbstractArticleService<E> {

    private ArticleCategoryService categoryService;

    private ArticleCategoryMappingService mappingService;

    @Override
    public final Collection<E> queryAll(ArticleSearchParams searchParams) {
        if (interceptQuery(searchParams)) {
            return Collections.emptyList();
        }

        return selectAll(searchParams);
    }

    @Override
    public final ListObject<E> queryPage(ArticleSearchParams searchParams) {
        if (interceptQuery(searchParams)) {
            return ListObject.empty();
        }

        return selectPage(searchParams);
    }

    @Override
    public final int queryCount(ArticleSearchParams searchParams) {
        if (interceptQuery(searchParams)) {
            return 0;
        }

        return selectCount(searchParams);
    }

    private boolean interceptQuery(ArticleSearchParams searchParams) {
        Collection<Long> categoryIds = findCategoryIds(searchParams);

        if (categoryIds == null) {
            return false;
        }
        if (categoryIds.isEmpty()) {
            return true;
        }

        ArticleCategoryMappingSearchParams mappingSearchParams = new ArticleCategoryMappingSearchParams();
        mappingSearchParams.setArticleCategoryIds(categoryIds);
        mappingSearchParams.setArticleIds(searchParams.getArticleIds());

        Collection<Long> articleIds = CollectionUtil.toList(getMappingService().findList(mappingSearchParams), ArticleCategoryMapping::getArticleId);
        searchParams.setArticleIds(articleIds);

        return articleIds.isEmpty();
    }

    private Collection<Long> findCategoryIds(ArticleSearchParams searchParams) {
        Long articleCategoryId = searchParams.getArticleCategoryId();
        String articleCategoryType = StringUtils.trimToNull(searchParams.getArticleCategoryType());

        searchParams.setArticleCategoryId(null);
        searchParams.setArticleCategoryType(null);

        if (!NumberHelper.greaterZero(articleCategoryId) && articleCategoryType == null) {
            return null;
        }

        ArticleCategorySearchParams categorySearchParams = new ArticleCategorySearchParams();
        categorySearchParams.setArticleCategoryId(articleCategoryId);
        categorySearchParams.setArticleCategoryType(articleCategoryType);

        return CollectionUtil.toList(getCategoryService().selectAll(categorySearchParams), ArticleCategory::getArticleCategoryId);
    }

    protected ArticleCategoryService getCategoryService() {
        return categoryService;
    }

    protected ArticleCategoryMappingService getMappingService() {
        return mappingService;
    }

    @Autowired
    public void setCategoryService(ArticleCategoryService categoryService) {
        this.categoryService = categoryService;
    }

    @Autowired
    public void setMappingService(ArticleCategoryMappingService mappingService) {
        this.mappingService = mappingService;
    }
}
