/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.article.service.service;

import net.guerlab.smart.article.core.domain.ArticleDTO;
import net.guerlab.smart.article.core.searchparams.ArticleSearchParams;
import net.guerlab.smart.platform.server.service.BaseFindService;
import net.guerlab.spring.commons.dto.Convert;
import net.guerlab.web.result.ListObject;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;

/**
 * 抽象文章服务
 *
 * @param <E>
 *         文章类型
 * @author guer
 */
public interface AbstractArticleService<E extends Convert<ArticleDTO>> extends BaseFindService<E, Long, ArticleSearchParams> {

    /**
     * 数值正则表达式
     */
    String NUMBER_REG = "^(-?\\d*)$";

    /**
     * 标题最大长度
     */
    int TITLE_MAX_LENGTH = 255;

    /**
     * 唯一key最大长度
     */
    int UNIQUE_KEY_MAX_LENGTH = 50;

    /**
     * 简介最大长度
     */
    int SYNOPSIS_MAX_LENGTH = 255;

    /**
     * 原文链接最大长度
     */
    int ORIGINAL_LINK_MAX_LENGTH = 255;

    /**
     * 作者最大长度
     */
    int AUTHOR_MAX_LENGTH = 100;

    /**
     * keywords最大长度
     */
    int KEYWORDS_MAX_LENGTH = 255;

    /**
     * description最大长度
     */
    int DESCRIPTION_MAX_LENGTH = 255;

    /**
     * 获取所有对象
     *
     * @param searchParams
     *         搜索参数对象
     * @return 实体列表
     */
    Collection<E> queryAll(ArticleSearchParams searchParams);

    /**
     * 查询列表
     *
     * @param searchParams
     *         搜索参数对象
     * @return 实体列表
     */
    ListObject<E> queryPage(ArticleSearchParams searchParams);

    /**
     * 查询总记录数
     *
     * @param searchParams
     *         搜索参数对象
     * @return 实体总数
     */
    int queryCount(ArticleSearchParams searchParams);

    /**
     * 通过唯一key查询文章
     *
     * @param uniqueKey
     *         唯一key
     * @return 文章
     */
    default E selectByUniqueKey(String uniqueKey) {
        String key = StringUtils.trimToNull(uniqueKey);
        if (key == null) {
            return null;
        }

        ArticleSearchParams searchParams = new ArticleSearchParams();
        searchParams.setUniqueKey(key);

        return selectOne(searchParams);
    }
}
