package net.guerlab.smart.article.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.article.core.enums.PublishType;
import net.guerlab.smart.article.core.exception.*;
import net.guerlab.smart.article.core.searchparams.ArticleSearchParams;
import net.guerlab.smart.article.service.entity.Article;
import net.guerlab.smart.article.service.entity.ArticleCategory;
import net.guerlab.smart.article.service.mapper.ArticleMapper;
import net.guerlab.smart.article.service.service.ArticleCategoryService;
import net.guerlab.smart.article.service.service.ArticleCategoryUpdateHandler;
import net.guerlab.smart.article.service.service.ArticleService;
import net.guerlab.smart.platform.commons.util.OrderEntityUtils;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.regex.Pattern;

/**
 * 文章服务实现
 *
 * @author guer
 */
@Service
public class ArticleServiceImpl extends BaseServiceImpl<Article, Long, ArticleMapper>
        implements ArticleService, ArticleCategoryUpdateHandler {

    private ArticleCategoryService categoryService;

    @Override
    public void updateArticleCategory(ArticleCategory articleCategory) {
        Long categoryId = articleCategory.getArticleCategoryId();
        String categoryName = StringUtils.trimToNull(articleCategory.getArticleCategoryName());
        String categoryType = StringUtils.trimToNull(articleCategory.getArticleCategoryType());

        if (refuseUpdateArticleCategory(categoryId, categoryName, categoryType)) {
            return;
        }

        Article entity = new Article();
        entity.setArticleCategoryName(categoryName);
        entity.setArticleCategoryType(categoryType);

        ArticleSearchParams searchParams = new ArticleSearchParams();
        searchParams.setArticleCategoryId(categoryId);

        mapper.updateByExampleSelective(entity, getExample(searchParams));
    }

    private boolean refuseUpdateArticleCategory(Long categoryId, String categoryName, String categoryType) {
        return !NumberHelper.greaterZero(categoryId) || (categoryName == null && categoryType == null);
    }

    @Override
    protected void insertBefore(Article entity) {
        String title = StringUtils.trimToNull(entity.getTitle());
        if (title == null) {
            throw new ArticleTitleInvalidException();
        } else if (title.length() > TITLE_MAX_LENGTH) {
            throw new ArticleTitleLengthErrorException();
        }
        entity.setTitle(title);

        setAuthor(entity);
        setSynopsis(entity);
        setOriginalLink(entity);
        setUniqueKey(entity);

        ArticleCategory articleCategory = findArticleCategory(entity.getArticleCategoryId());
        entity.setArticleCategoryName(articleCategory.getArticleCategoryName());
        entity.setArticleCategoryType(articleCategory.getArticleCategoryType());

        LocalDateTime now = LocalDateTime.now();
        if (entity.getReleaseTime() == null) {
            entity.setReleaseTime(now);
        }
        if (entity.getAlwaysRedirect() == null) {
            entity.setAlwaysRedirect(false);
        }

        PublishType publishType = entity.getPublishType();
        if (publishType == null) {
            publishType = PublishType.AUTOMATIC;
        }
        entity.setPublishType(publishType);
        setPublishType(entity, publishType, now);

        entity.setArticleId(sequence.nextId());
        entity.setCreateTime(now);
        entity.setUpdateTime(now);
        OrderEntityUtils.propertiesCheck(entity);
    }

    private void setUniqueKey(Article entity) {
        String uniqueKey = StringUtils.trimToNull(entity.getUniqueKey());
        if (uniqueKey != null) {
            if (uniqueKey.length() > UNIQUE_KEY_MAX_LENGTH) {
                throw new UniqueKeyLengthErrorException();
            } else if (Pattern.matches(NUMBER_REG, uniqueKey)) {
                throw new UniqueKeyFormatErrorException();
            } else if (selectByUniqueKey(uniqueKey) != null) {
                throw new UniqueKeyRepeatException();
            }
        } else {
            uniqueKey = "";
        }
        entity.setUniqueKey(uniqueKey);
    }

    @Override
    protected void updateBefore(Article entity) {
        String title = StringUtils.trimToNull(entity.getTitle());
        if (title != null && title.length() > TITLE_MAX_LENGTH) {
            throw new ArticleTitleLengthErrorException();
        }
        entity.setTitle(title);

        setAuthor(entity);
        setSynopsis(entity);
        setOriginalLink(entity);

        Long articleCategoryId = entity.getArticleCategoryId();
        if (NumberHelper.greaterZero(articleCategoryId)) {
            ArticleCategory articleCategory = findArticleCategory(articleCategoryId);
            entity.setArticleCategoryName(articleCategory.getArticleCategoryName());
            entity.setArticleCategoryType(articleCategory.getArticleCategoryType());
        } else {
            entity.setArticleCategoryId(null);
            entity.setArticleCategoryName(null);
            entity.setArticleCategoryType(null);
        }

        PublishType publishType = entity.getPublishType();
        LocalDateTime now = LocalDateTime.now();
        if (publishType != null) {
            setPublishType(entity, publishType, now);
        }

        entity.setUpdateTime(now);
    }

    private void setAuthor(Article entity) {
        String author = StringUtils.trimToNull(entity.getAuthor());
        if (author != null && author.length() > AUTHOR_MAX_LENGTH) {
            throw new AuthorLengthErrorException();
        }
        entity.setAuthor(author);
    }

    private void setSynopsis(Article entity) {
        String synopsis = StringUtils.trimToNull(entity.getSynopsis());
        if (synopsis != null && synopsis.length() > SYNOPSIS_MAX_LENGTH) {
            throw new SynopsisLengthErrorException();
        }
        entity.setSynopsis(synopsis);
    }

    private void setOriginalLink(Article entity) {
        String originalLink = StringUtils.trimToNull(entity.getOriginalLink());
        if (originalLink != null && originalLink.length() > ORIGINAL_LINK_MAX_LENGTH) {
            throw new OriginalLinkLengthErrorException();
        }
        entity.setOriginalLink(originalLink);
    }

    private void setPublishType(Article entity, PublishType publishType, LocalDateTime now) {
        switch (publishType) {
            case AUTOMATIC:
                entity.setPlanPublishTime(now);
                entity.setPublishTime(now);
                entity.setPublished(true);
                break;
            case TIMING:
                LocalDateTime planPublishTime = entity.getPlanPublishTime();
                if (planPublishTime == null) {
                    throw new PlanPublishTimeInvalidException();
                } else if (planPublishTime.isBefore(now)) {
                    entity.setPublished(true);
                    entity.setPublishTime(now);
                } else {
                    entity.setPublishTime(null);
                    entity.setPublished(false);
                }
                break;
            case MANUAL:
                if (entity.getPublished() == null) {
                    entity.setPlanPublishTime(null);
                    entity.setPublishTime(null);
                } else if (entity.getPublished()) {
                    entity.setPlanPublishTime(now);
                    entity.setPublishTime(now);
                } else {
                    entity.setPlanPublishTime(now);
                    entity.setPublishTime(now);
                }
                break;
            default:
        }
    }

    private ArticleCategory findArticleCategory(Long articleCategoryId) {
        if (!NumberHelper.greaterZero(articleCategoryId)) {
            throw new ArticleCategoryIdInvalidException();
        }

        return categoryService.selectByIdOptional(articleCategoryId).orElseThrow(ArticleCategoryInvalidException::new);
    }

    @Autowired
    public void setCategoryService(ArticleCategoryService categoryService) {
        this.categoryService = categoryService;
    }
}
