package net.guerlab.smart.article.service.service.impl;

import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.article.core.exception.ArticleCategoryDisallowedDeleteException;
import net.guerlab.smart.article.core.exception.ArticleCategoryNameInvalidException;
import net.guerlab.smart.article.core.exception.ArticleCategoryNameLengthErrorException;
import net.guerlab.smart.article.core.exception.ParentArticleCategoryInvalidException;
import net.guerlab.smart.article.core.searchparams.ArticleCategorySearchParams;
import net.guerlab.smart.article.core.searchparams.ArticleSearchParams;
import net.guerlab.smart.article.service.entity.ArticleCategory;
import net.guerlab.smart.article.service.mapper.ArticleCategoryMapper;
import net.guerlab.smart.article.service.service.ArticleCategoryService;
import net.guerlab.smart.article.service.service.ArticleCategoryUpdateHandler;
import net.guerlab.smart.article.service.service.ArticleService;
import net.guerlab.smart.platform.commons.Constants;
import net.guerlab.smart.platform.commons.util.OrderEntityUtils;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.spring.commons.util.SpringApplicationContextUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Optional;

/**
 * 文章分类服务实现
 *
 * @author guer
 */
@Service
public class ArticleCategoryServiceImpl extends BaseServiceImpl<ArticleCategory, Long, ArticleCategoryMapper>
        implements ArticleCategoryService {

    private ArticleService articleService;

    @Override
    protected void insertBefore(ArticleCategory entity) {
        String name = StringUtils.trimToNull(entity.getArticleCategoryName());
        String type = StringUtils.trimToNull(entity.getArticleCategoryType());
        Long parentId = entity.getParentId();

        if (name == null) {
            throw new ArticleCategoryNameInvalidException();
        } else if (name.length() > CATEGORY_NAME_MAX_LENGTH) {
            throw new ArticleCategoryNameLengthErrorException();
        }

        if (NumberHelper.greaterZero(parentId)) {
            if (selectById(parentId) == null) {
                throw new ParentArticleCategoryInvalidException();
            }
        } else {
            entity.setParentId(Constants.DEFAULT_PARENT_ID);
        }

        entity.setArticleCategoryId(sequence.nextId());
        entity.setArticleCategoryName(name);
        entity.setArticleCategoryType(Optional.ofNullable(type).orElse(ArticleCategory.DEFAULT_ARTICLE_CATEGORY_TYPE));
        if (entity.getEnabled() == null) {
            entity.setEnabled(true);
        }

        OrderEntityUtils.propertiesCheck(entity);
    }

    @Override
    protected void updateBefore(ArticleCategory entity) {
        String name = StringUtils.trimToNull(entity.getArticleCategoryName());

        if (name != null && name.length() > CATEGORY_NAME_MAX_LENGTH) {
            throw new ArticleCategoryNameLengthErrorException();
        }

        entity.setArticleCategoryName(name);
    }

    @Override
    protected void updateAfter(ArticleCategory entity) {
        SpringApplicationContextUtil.getContext().getBeansOfType(ArticleCategoryUpdateHandler.class).values()
                .forEach(handler -> handler.updateArticleCategory(entity));
    }

    @Override
    protected void deleteBefore(ArticleCategory entity, Boolean force) {
        deleteByIdBefore(entity.getArticleCategoryId(), force);
    }

    @Override
    protected void deleteByIdBefore(Long id, Boolean force) {
        if (!NumberHelper.greaterZero(id)) {
            return;
        }

        ArticleCategorySearchParams articleCategorySearchParams = new ArticleCategorySearchParams();
        articleCategorySearchParams.setParentId(id);

        ArticleSearchParams articleSearchParams = new ArticleSearchParams();
        articleSearchParams.setArticleCategoryId(id);

        if (selectCount(articleCategorySearchParams) > 0 || articleService.selectCount(articleSearchParams) > 0) {
            throw new ArticleCategoryDisallowedDeleteException();
        }
    }

    @Autowired
    public void setArticleService(ArticleService articleService) {
        this.articleService = articleService;
    }
}
