package net.groupadd.yena.common.configuration.core;

import net.groupadd.yena.common.configuration.api.Source;
import net.groupadd.yena.common.configuration.api.SourceLoader;
import net.groupadd.yena.common.configuration.api.SourceLoaderFactory;
import net.groupadd.yena.common.configuration.enums.Ext;
import net.groupadd.yena.common.configuration.exceptions.UnsupportedExtensionException;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * The type Source.
 */
class SourceImpl implements Source {

    private final SourceLoaderFactory sourceLoaderFactory;
    private final Set<String> pathSet;

    private final Map<String, Object> sourceMap;

    private final Lock readLock;
    private final Lock writeLock;


    /**
     * Instantiates a new Source.
     *
     * @param sourceMap           the source map
     * @param sourceLoaderFactory the source loader factory
     * @param pathSet             the path set
     */
    SourceImpl(final Map<String, Object> sourceMap, final SourceLoaderFactory sourceLoaderFactory, final Set<String> pathSet) {
        this.sourceLoaderFactory = sourceLoaderFactory;
        this.pathSet = pathSet;
        this.sourceMap = sourceMap;
        final ReentrantReadWriteLock reentrantReadWriteLock = new ReentrantReadWriteLock();
        this.readLock = reentrantReadWriteLock.readLock();
        this.writeLock = reentrantReadWriteLock.writeLock();
    }


    private Ext getExtension(String path) {
        int pos = path.lastIndexOf(DOT);
        if (pos != -1) {
            return Ext.valueOf(path.substring(pos + 1).toUpperCase());
        }
        throw new UnsupportedExtensionException("Unable to get a proper extension with : " + path);
    }

    @Override
    public void init() {
        this.writeLock.lock();
        try {
            this.sourceMap.clear();
            pathSet.stream().map(s -> {
                final Ext ext = getExtension(s);
                final SourceLoader loader = sourceLoaderFactory.create(ext);
                return loader.load(s);
            }).forEach(map -> {
                this.sourceMap.putAll(map);
            });
        } finally {
            this.writeLock.unlock();
        }

    }

    @Override
    public Object get(String key) {
        this.readLock.lock();
        try {
            Object val = getPropsValue(this.sourceMap, key);
            if (val == null) {
                val = getYmlValue(this.sourceMap, key);
            }
            return val;
        } finally {
            this.readLock.unlock();
        }
    }

    private Object getPropsValue(Map<String, Object> map, String key) {
        return map.get(key);
    }

    private Object getYmlValue(Map<String, Object> map, String key) {
        if (key == null) {
            return null;
        }

        Map<String, Object> newMap = map;
        Object res = null;

        if (key.length() - 1 == key.lastIndexOf(DOT)) {
            return null;
        }

        while (key.length() > 0) {
            final int pos = key.indexOf(DOT);
            if (pos == -1) {
                res = newMap.get(key);
                break;
            } else if (pos == 0) {
                res = null;
                break;
            } else {
                final String k = key.substring(0, pos);
                final Object obj = newMap.get(k);
                if (obj instanceof LinkedHashMap) {
                    newMap = (Map<String, Object>) obj;
                    key = key.substring(pos + 1);
                } else {
                    res = obj;
                    key = key.substring(pos + 1);
                }
            }
        }
        return res;
    }

    @Override
    public void clear() {
        this.writeLock.lock();
        try {
            this.sourceMap.clear();
        } finally {
            this.writeLock.unlock();
        }
    }
}
