package net.groupadd.yena.common.configuration.core;

import net.groupadd.yena.common.configuration.api.Source;
import net.groupadd.yena.common.configuration.api.SourceLoaderFactory;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * The type Source builder.
 */
public class SourceBuilder {

    private final int delay;
    private final TimeUnit timeUnit;

    private final Map<String, Object> map;
    private final SourceLoaderFactory factory;
    private final Set<String> paths;

    private SourceBuilder(final Builder builder) {
        this.delay = builder.delay;
        this.timeUnit = builder.timeUnit;
        this.paths = builder.paths;
        this.factory = new SourceLoaderFactoryImpl();
        this.map = new HashMap<>();
    }

    /**
     * Source source.
     *
     * @return the source
     */
    public Source source() {
        return new SourceImpl(this.map, this.factory, this.paths);
    }

    /**
     * Delay int.
     *
     * @return the int
     */
    public int delay() {
        return delay;
    }

    /**
     * Time unit time unit.
     *
     * @return the time unit
     */
    public TimeUnit timeUnit() {
        return timeUnit;
    }

    /**
     * Builder builder.
     *
     * @return the builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * The type Builder.
     */
    public static class Builder {

        private Set<String> paths;
        private int delay;
        private TimeUnit timeUnit;

        /**
         * Instantiates a new Builder.
         */
        Builder() {
            this.paths = new HashSet<>();
            this.delay = 10;
            this.timeUnit = TimeUnit.MINUTES;
        }

        /**
         * Path builder.
         *
         * @param path the path
         * @return the builder
         */
        public Builder path(String path) {
            this.paths.add(path);
            return this;
        }

        /**
         * Paths builder.
         *
         * @param paths the paths
         * @return the builder
         */
        public Builder paths(Set<String> paths) {
            this.paths.addAll(paths);
            return this;
        }

        /**
         * Delay builder.
         *
         * @param delay the delay
         * @return the builder
         */
        public Builder delay(int delay) {
            this.delay = delay;
            return this;
        }

        /**
         * Time unit builder.
         *
         * @param timeUnit the time unit
         * @return the builder
         */
        public Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        /**
         * Build source builder.
         *
         * @return the source builder
         */
        public SourceBuilder build() {
            return new SourceBuilder(this);
        }

    }
}
